/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/httptask.h"

#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/writer.h"

namespace {
const int kDefaultRequestTimeoutInSec = 10;
}

ttv::HttpTask::HttpRequestInfo::HttpRequestInfo()
    : httpReqType(HTTP_INVALID_REQUEST), timeOutInSecs(kDefaultRequestTimeoutInSec) {}

ttv::HttpTask::HttpTask() : Task(nullptr, nullptr) {}

ttv::HttpTask::HttpTask(const std::string& authToken) : Task(nullptr, nullptr), mAuthToken(authToken) {}

ttv::HttpTask::HttpTask(TaskCallback callback, void* userData, const char* authToken) : Task(callback, userData) {
  if (authToken != nullptr && authToken[0] != '\0') {
    mAuthToken = authToken;
  }
}

void ttv::HttpTask::Run() {
  if (!mAborted) {
    FillHttpRequestInfo(mHttpRequestInfo);

    // Add the OAuth token if configured
    if (mAuthToken.size() > 0) {
      if (!ContainsHttpParameter(mHttpRequestInfo.requestHeaders, "Authorization")) {
        mHttpRequestInfo.requestHeaders.emplace_back(HttpParam("Authorization", "OAuth " + mAuthToken));
      }
    }

    // Prefer to receive gzipped responses if no other specified
    if (!ContainsHttpParameter(mHttpRequestInfo.requestHeaders, "Accept-Encoding")) {
      mHttpRequestInfo.requestHeaders.emplace_back(HttpParam("Accept-Encoding", "gzip"));
    }

    std::function<bool(uint, const std::map<std::string, std::string>&, void*)> headersCallback =
      [this](uint statusCode, const std::map<std::string, std::string>& headers, void* /*userData*/) {
        return ProcessHeaders(statusCode, headers);
      };

    std::function<void(uint, const std::vector<char>&, void*)> responseCallback =
      [this](uint statusCode, const std::vector<char>& response, void* /*userData*/) {
        return ResponseCallback(statusCode, response);
      };

    using namespace std::placeholders;
    TTV_ErrorCode ec = SendHttpRequest(GetTaskName(), mHttpRequestInfo.url, mHttpRequestInfo.requestHeaders,
      reinterpret_cast<const uint8_t*>(mHttpRequestInfo.requestBody.c_str()), mHttpRequestInfo.requestBody.size(),
      mHttpRequestInfo.httpReqType, mHttpRequestInfo.timeOutInSecs, 0, headersCallback, responseCallback, nullptr);

    // if the request attempt itself failed then return that error
    if (TTV_FAILED(ec)) {
      PopulateErrorDetails(ec, 0, "SendHttpRequest synchronous failure");
    }
  } else {
    PopulateErrorDetails(TTV_EC_REQUEST_ABORTED, 0, "Task aborted");
  }
}

void ttv::HttpTask::OnComplete() {
  if (mTaskCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mTaskCallback(mTaskStatus.ec, mUserData);
  }
}

bool ttv::HttpTask::HeadersCallback(uint statusCode, const std::map<std::string, std::string>& headers) {
  bool wantResponse = false;

  json::FastWriter writer;

  if (!mAborted) {
    if (statusCode >= 200 && statusCode < 300) {
      wantResponse = ProcessHeaders(statusCode, headers);
    } else if (statusCode == 401) {
      PopulateErrorDetails(TTV_EC_AUTHENTICATION, statusCode, "Authentication error");
    } else {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "HTTP request failed with status code %d.", statusCode);
      PopulateErrorDetails(TTV_EC_API_REQUEST_FAILED, statusCode, "Request failed");
    }
  } else {
    PopulateErrorDetails(TTV_EC_REQUEST_ABORTED, statusCode, "Task aborted");
  }

  return wantResponse;
}

void ttv::HttpTask::ResponseCallback(uint statusCode, const std::vector<char>& response) {
  if (!mAborted) {
    if (statusCode >= 200 && statusCode < 300) {
      ProcessResponse(statusCode, response);
    } else if (statusCode == 401) {
      PopulateErrorDetails(TTV_EC_AUTHENTICATION, statusCode, "Authentication error");
    } else {
      std::string msg(response.data(), response.size());
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "HTTP request failed with status code %d. Message: %s",
        statusCode, msg.c_str());
      PopulateErrorDetails(TTV_EC_API_REQUEST_FAILED, statusCode, "Request failed");
    }
  } else {
    PopulateErrorDetails(TTV_EC_REQUEST_ABORTED, statusCode, "Task aborted");
  }
}

bool ttv::HttpTask::ProcessHeaders(uint /*statusCode*/, const std::map<std::string, std::string>& /*headers*/) {
  return true;
}

void ttv::HttpTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& /*response*/) {}

void ttv::HttpTask::PopulateErrorDetails(TTV_ErrorCode ec, uint statusCode, const std::string& message) {
  mTaskStatus.ec = ec;
  mTaskStatus["Message"] = message;
  mTaskStatus["RequestName"] = GetTaskName();
  mTaskStatus["StatusCode"] = statusCode;
  mTaskStatus["Url"] = mHttpRequestInfo.url;
}
