/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/simplejsonhttptask.h"

#include "twitchsdk/core/json/reader.h"

ttv::SimpleJsonHttpTask::SimpleJsonHttpTask(
  const std::string& url, HttpRequestType method, const std::string& authToken, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mUrl(url), mMethod(method), mNumRetries(0), mCallback(callback) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SimpleJsonHttpTask created");
}

void ttv::SimpleJsonHttpTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = mUrl;
  requestInfo.httpReqType = mMethod;
}

void ttv::SimpleJsonHttpTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  mResult = std::make_shared<Result>();

  if (response.size() > 0) {
    // Parse the returned JSON
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), mResult->json);
    if (!parsed) {
      ttv::trace::Message(
        "SimpleJsonHttpTask", MessageLevel::Error, "Inside SimpleJsonHttpTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      mResult.reset();
    }
  } else {
    mResult->json = json::nullValue;
  }
}

void ttv::SimpleJsonHttpTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
