/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/taskrunner.h"

#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/result.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/thread.h"

ttv::TaskRunner::TaskRunner() : mName("Default"), mState(TaskRunnerState::ShutDown) {
  Initialize();
}

ttv::TaskRunner::TaskRunner(const std::string& name) : mName(name), mState(TaskRunnerState::ShutDown) {
  Initialize();
}

ttv::TaskRunner::~TaskRunner() {
  mBackgroundEventScheduler.reset();

  ttv::trace::Message("TaskRunner", MessageLevel::Info, "TaskRunner destroyed");
}

void ttv::TaskRunner::Initialize() {
  TTV_ErrorCode ec = CreateBackgroundEventScheduler(
    mName + "-TaskRunner(" + ttv::PointerToString(this) + ")", mBackgroundEventScheduler);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mBackgroundEventScheduler != nullptr);

  mState = TaskRunnerState::Running;

  // TODO: https://jira.twitch.com/browse/SDK-175
  // Remove this once we refactor HTTP tasks.
  mBackgroundEventScheduler->ScheduleTask({[] {
    TTV_ErrorCode ret = HttpThreadInit();
    ASSERT_ON_ERROR(ret);
  }});

  ttv::trace::Message("TaskRunner", MessageLevel::Info, "TaskRunner [%s] created", mName.c_str());
}

bool ttv::TaskRunner::AddTask(std::shared_ptr<Task> task) {
  TTV_ASSERT(task != nullptr);
  if (task == nullptr) {
    ttv::trace::Message(
      "TaskRunner", MessageLevel::Debug, "TaskRunner::AddTask() [%s] - null task added", mName.c_str());
    return false;
  }

  // make sure not shutdown
  if (mState != TaskRunnerState::Running) {
    ttv::trace::Message("TaskRunner", MessageLevel::Debug,
      "TaskRunner::AddTask() [%s] - Task added while shutting down: 0x%x - %s", mName.c_str(), task.get(),
      task->GetTaskName());
    return false;
  }

  Result<TaskId> result = mBackgroundEventScheduler->ScheduleTask({[this, task] {
    task->Run();
    mDoneTaskQ.push(task);
  }});

  TTV_ASSERT(!result.IsError());

  return true;
}

void ttv::TaskRunner::PollTasks() {
  std::shared_ptr<Task> task;
  while (mDoneTaskQ.try_pop(task)) {
    ttv::trace::Message("TaskRunner", MessageLevel::Debug, "TaskRunner::PollTasks() [%s] - Completing task: 0x%x - %s",
      mName.c_str(), task.get(), task->GetTaskName());

    task->OnComplete();
  }
}

TTV_ErrorCode ttv::TaskRunner::Shutdown() {
  TaskRunnerState expected = TaskRunnerState::Running;

  if (!mState.compare_exchange_strong(expected, TaskRunnerState::ShuttingDown)) {
    return TTV_EC_NOT_INITIALIZED;
  }

  ttv::trace::Message("TaskRunner", MessageLevel::Info, "TaskRunner [%s] shutting down", mName.c_str());

  // TODO: https://jira.twitch.com/browse/SDK-175
  // Remove this once we refactor HTTP tasks.
  mBackgroundEventScheduler->ScheduleTask({[] {
    TTV_ErrorCode ret = HttpThreadShutdown();
    ASSERT_ON_ERROR(ret);
  }});

  return mBackgroundEventScheduler->Shutdown([this]() { mState = TaskRunnerState::ShutDown; });
}

void ttv::TaskRunner::CompleteShutdown() {
  TTV_ASSERT(mState == TaskRunnerState::ShutDown);
  PollTasks();

  // polling tasks will flush the done queue
  TTV_ASSERT(mDoneTaskQ.empty());

  ttv::trace::Message("TaskRunner", MessageLevel::Info, "TaskRunner [%s] shutdown complete", mName.c_str());
}
