/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/validateoauthtask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"

#include <sstream>

// https://github.com/justintv/Twitch-API/blob/master/v2_resources/root.md

// curl -H "Client-Id: $CLIENTID" -X GET "https://api.twitch.tv/kraken?oauth_token=$OAUTH" | python -m json.tool
// GET /kraken/

namespace {
const char* kAuthStatusURL = "https://api.twitch.tv/kraken";
}

ttv::ValidateOAuthTask::ValidateOAuthTask(const std::string& oauthToken, Callback callback)
    : mOAuthToken(oauthToken), mCallback(callback) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Debug, "AuthStatusTask ctor");
}

void ttv::ValidateOAuthTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri url;
  url.SetUrl(kAuthStatusURL);
  url.SetParam("oauth_token", mOAuthToken);

  requestInfo.url = url;
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::ValidateOAuthTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jToken = jsonVal["token"];
    if (jToken.isNull() || !jToken.isObject()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jValid = jToken["valid"];
    if (jValid.isNull() || !jValid.isBool()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();
    mResult->valid = jValid.asBool();

    if (!mResult->valid) {
      return;
    }

    const auto& jUserName = jToken["user_name"];
    if (jUserName.isNull() || !jUserName.isString()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult->userName = jUserName.asString();

    const auto& jAuthorization = jToken["authorization"];
    if (jAuthorization.isNull() || !jAuthorization.isObject()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jScopes = jAuthorization["scopes"];
    if (jScopes.isNull() || !jScopes.isArray()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    for (size_t i = 0; i < jScopes.size(); ++i) {
      mResult->scopes.push_back(jScopes[i].asString());
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::ValidateOAuthTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
