/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/threadedeventscheduler.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"

namespace {
const char* kLoggerName = "ThreadedEventScheduler";
}

ttv::ThreadedEventScheduler::ThreadedEventScheduler()
    : mEventQueue("ThreadedEventScheduler"),
      mName("ThreadedEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::ShutDown) {
  StartThreadProc();
}

ttv::ThreadedEventScheduler::ThreadedEventScheduler(const std::string& name)
    : mEventQueue("ThreadedEventScheduler-" + name),
      mName(name + "-ThreadedEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::ShutDown) {
  StartThreadProc();
}

ttv::ThreadedEventScheduler::~ThreadedEventScheduler() {
  TTV_ASSERT(mState == EventSchedulerState::ShutDown);
  if (mState != EventSchedulerState::ShutDown) {
    Shutdown(nullptr);
    mEventSchedulerThread->Join();
    ttv::trace::Message(kLoggerName, MessageLevel::Debug,
      "Trying to destroy ThreadedEventScheduler \"%s\" that is still running", mName.c_str());
  }

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ThreadedEventScheduler \"%s\" destroyed", mName.c_str());
}

void ttv::ThreadedEventScheduler::StartThreadProc() {
  TTV_ASSERT(mState == EventSchedulerState::ShutDown);

  TTV_ErrorCode ec =
    ttv::CreateThread(std::bind(&ThreadedEventScheduler::EventSchedulerThreadProc, this), mName, mEventSchedulerThread);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mEventSchedulerThread != nullptr);

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ThreadedEventScheduler \"%s\" created", mName.c_str());

  mState = EventSchedulerState::Running;
  mEventSchedulerThread->Run();
}

void ttv::ThreadedEventScheduler::EventSchedulerThreadProc() {
  while (mState != EventSchedulerState::ShutDown) {
    mEventQueue.WaitForEvent();
  }

  mEventQueue.Clear();

  if (mShutDownTask != nullptr) {
    mShutDownTask();
    mShutDownTask = nullptr;
  }

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ThreadedEventScheduler \"%s\" shutdown", mName.c_str());
}

ttv::Result<ttv::TaskId> ttv::ThreadedEventScheduler::ScheduleTask(TaskParams&& taskParams) {
  if (mState != EventSchedulerState::Running) {
    return MakeErrorResult(TTV_EC_NOT_INITIALIZED);
  }

  return MakeSuccessResult(mEventQueue.InsertTask(std::move(taskParams)));
}

TTV_ErrorCode ttv::ThreadedEventScheduler::CancelTask(TaskId taskId) {
  if (mState != EventSchedulerState::Running) {
    return TTV_EC_NOT_INITIALIZED;
  }

  bool removed = mEventQueue.RemoveTask(taskId);

  if (!removed) {
    return TTV_EC_OPERATION_FAILED;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ThreadedEventScheduler::Shutdown(TaskFunc&& shutdownTask) {
  EventSchedulerState expected = EventSchedulerState::Running;
  if (!mState.compare_exchange_strong(expected, EventSchedulerState::ShuttingDown)) {
    return TTV_EC_NOT_INITIALIZED;
  }

  mEventQueue.InsertTask({[this, shutdownTask = std::move(shutdownTask)]() {
                            TTV_ASSERT(mState == EventSchedulerState::ShuttingDown);
                            EventSchedulerState expectedState = EventSchedulerState::ShuttingDown;
                            if (mState.compare_exchange_strong(expectedState, EventSchedulerState::ShutDown)) {
                              mShutDownTask = shutdownTask;
                            }
                          },
    "Shutting down ThreadedEventScheduler"});

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ThreadedEventScheduler \"%s\" shutting down", mName.c_str());

  return TTV_EC_SUCCESS;
}

ttv::IEventScheduler::EventSchedulerState ttv::ThreadedEventScheduler::GetState() {
  return mState;
}
