/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/
#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/tracer.h"

#include "twitchsdk/core/mutex.h"

#include <codecvt>
#include <ctime>
#include <locale>

namespace {
using namespace ttv;

std::shared_ptr<ITracer> gTracer;

const char* MessageLevelToString(const MessageLevel messageLevel) {
  const char* ret;

  switch (messageLevel) {
    case MessageLevel::Debug:
      ret = "DBG";
      break;
    case MessageLevel::Info:
      ret = "INF";
      break;
    case MessageLevel::Warning:
      ret = "WRN";
      break;
    case MessageLevel::Error:
      ret = "ERR";
      break;
    case MessageLevel::None:
    default:
      ret = "Invalid message level";
      break;
  }

  return ret;
}
}  // namespace

ttv::TracerBase::TracerBase() : mGlobalTraceLevel(MessageLevel::Error), mTraceFile(nullptr) {
  TTV_ErrorCode ret = ttv::CreateMutex(mMutex, "TracerBase");
  ASSERT_ON_ERROR(ret);
  mPrintBuffer.resize(1024);
}

ttv::TracerBase::~TracerBase() {
  CloseFile();
}

void ttv::TracerBase::Message(const char* component, const MessageLevel messageLevel, const char* format, ...) {
  if (!ShouldLog(component, messageLevel)) {
    return;
  }

  va_list args;

  va_start(args, format);

  {
    ttv::AutoMutex lock(mMutex.get());

    Log(component, MessageLevelToString(messageLevel), format, args);
  }

  // lint -esym(438,args) suppress "Last value assigned to variable 'args' (defined at line 77) not used"
  va_end(args);  // lint !e1924 Suppress "C-style cast"
}

void ttv::TracerBase::MessageVaList(
  const char* component, const MessageLevel messageLevel, const char* format, va_list args) {
  if (!ShouldLog(component, messageLevel)) {
    return;
  }

  ttv::AutoMutex lock(mMutex.get());

  Log(component, MessageLevelToString(messageLevel), format, args);
}

TTV_ErrorCode ttv::TracerBase::SetOutputFile(const std::string& path) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  ttv::AutoMutex lock(mMutex.get());

  CloseFile();

  if (path.length() > 0) {
    OpenFile(path);

    if (mTraceFile != nullptr) {
      fprintf(mTraceFile, ">>---------------------------- Trace Start ----------------------------<<\n");
    } else {
      printf(">> UNABLE TO OPEN TRACE FILE\n");
      ec = TTV_EC_CANNOT_OPEN_FILE;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::TracerBase::SetComponentMessageLevel(const char* component, MessageLevel level) {
  TTV_ASSERT(level <= MessageLevel::None);
  if (level <= MessageLevel::None) {
    if (component == nullptr) {
      mGlobalTraceLevel = level;
    } else {
      mTraceLevels[component] = level;
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::TracerBase::GetComponentMessageLevel(const char* component, MessageLevel& level) {
  level = MessageLevel::None;

  if (component == nullptr) {
    return TTV_EC_INVALID_ARG;
  } else {
    auto iter = mTraceLevels.find(component);
    if (iter != mTraceLevels.end()) {
      level = iter->second;
    }

    return TTV_EC_SUCCESS;
  }
}

TTV_ErrorCode ttv::TracerBase::SetGlobalMessageLevel(MessageLevel level) {
  TTV_ASSERT(level <= MessageLevel::None);
  if (level <= MessageLevel::None) {
    mGlobalTraceLevel = level;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::TracerBase::GetGlobalMessageLevel(MessageLevel& level) {
  level = mGlobalTraceLevel;

  return TTV_EC_SUCCESS;
}

bool ttv::TracerBase::ShouldLog(const char* component, const MessageLevel messageLevel) {
  TTV_ASSERT(component != nullptr);
  if (component == nullptr) {
    return false;
  }

  TTV_ASSERT(component[0] != '\0');
  if (component[0] == '\0') {
    return false;
  }

  MessageLevel filterLevel = mGlobalTraceLevel;

  auto it = mTraceLevels.find(component);
  if (it != mTraceLevels.end()) {
    filterLevel = it->second;
  }

  return messageLevel >= filterLevel;
}

void ttv::TracerBase::GetLinePrefix(
  const char* component, const char* messageLevel, char* buffer, size_t bufferLength) {
  const time_t currentTime = time(nullptr);
  const struct tm* localTime = localtime(&currentTime);

  const size_t dateAndTimeSize = static_cast<size_t>(32);
  char dateAndTime[dateAndTimeSize + 1];

  (void)strftime(dateAndTime, dateAndTimeSize + 1, "%c", localTime);

  snprintf(buffer, bufferLength - 1, "%s: %s - %s - ", messageLevel, dateAndTime, component);
  buffer[bufferLength - 1] = '\0';
}

bool ttv::TracerBase::OpenFile(const std::string& path) {
  if (mTraceFile != nullptr) {
    CloseFile();
  }

  mTraceFile = fopen(path.c_str(), "wt");

  return mTraceFile != nullptr;
}

void ttv::TracerBase::LogToFile(const char* message) {
  if (mTraceFile != nullptr) {
    (void)fprintf(mTraceFile, "%s\n", message);
    fflush(mTraceFile);
  }
}

bool ttv::TracerBase::CloseFile() {
  if (mTraceFile != nullptr) {
    fclose(mTraceFile);
    mTraceFile = nullptr;
  }

  return true;
}

void ttv::SetTracer(std::shared_ptr<ITracer> tracer) {
  gTracer = tracer;
}

void ttv::trace::Message(const char* component, const MessageLevel messageLevel, const char* format, ...) {
  if (gTracer == nullptr) {
    return;
  }

  va_list args;
  va_start(args, format);

  gTracer->MessageVaList(component, messageLevel, format, args);

  // lint -esym(438,args) suppress "Last value assigned to variable 'args' (defined at line 77) not used"
  va_end(args);  // lint !e1924 Suppress "C-style cast"
}

void ttv::trace::MessageVaList(
  const char* component, const MessageLevel messageLevel, const char* format, va_list args) {
  if (gTracer == nullptr) {
    return;
  }

  gTracer->MessageVaList(component, messageLevel, format, args);
}

TTV_ErrorCode ttv::trace::SetOutputFile(const std::string& path) {
  if (gTracer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gTracer->SetOutputFile(path);
}

TTV_ErrorCode ttv::trace::SetComponentMessageLevel(const char* component, MessageLevel level) {
  if (gTracer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gTracer->SetComponentMessageLevel(component, level);
}

TTV_ErrorCode ttv::trace::GetComponentMessageLevel(const char* component, MessageLevel& level) {
  if (gTracer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gTracer->GetComponentMessageLevel(component, level);
}

TTV_ErrorCode ttv::trace::SetGlobalMessageLevel(MessageLevel level) {
  if (gTracer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gTracer->SetGlobalMessageLevel(level);
}

TTV_ErrorCode ttv::trace::GetGlobalMessageLevel(MessageLevel& level) {
  if (gTracer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gTracer->GetGlobalMessageLevel(level);
}
