/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/trackingcontext.h"

#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/mutex.h"

ttv::TrackingContext::TrackingContext() : TrackingContext(nullptr) {}

ttv::TrackingContext::TrackingContext(const std::shared_ptr<TrackingContext>& parent) : mParentContext(parent) {
  TTV_ErrorCode ret = ttv::CreateMutex(mMutex, "TrackingContext");
  ASSERT_ON_ERROR(ret);
}

void ttv::TrackingContext::SetProperty(const std::string& key, const ttv::TrackingValue& value) {
  ttv::AutoMutex lock(mMutex.get());
  mProperties[key] = value;
}

void ttv::TrackingContext::SetProperties(std::initializer_list<std::pair<std::string, TrackingValue>> properties) {
  ttv::AutoMutex lock(mMutex.get());
  for (const auto& pair : properties) {
    mProperties[pair.first] = pair.second;
  }
}

TTV_ErrorCode ttv::TrackingContext::TrackEvent(
  const std::string& eventName, std::map<std::string, TrackingValue>&& properties) {
  {
    ttv::AutoMutex lock(mMutex.get());
    for (const auto& pair : mProperties) {
      properties.insert(pair);
    }
  }

  if (mParentContext != nullptr) {
    return mParentContext->TrackEvent(eventName, std::move(properties));
  } else {
    return ttv::TrackEvent(eventName, properties);
  }
}
