/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/types/coretypes.h"

ttv::UserInfo::UserInfo() : userId(0), createdTimestamp(0) {}

bool ttv::UserInfo::operator==(const ttv::UserInfo& other) const {
  return (userName == other.userName) && (displayName == other.displayName) && (bio == other.bio) &&
         (logoImageUrl == other.logoImageUrl) && (userId == other.userId) &&
         (createdTimestamp == other.createdTimestamp);
}

bool ttv::UserInfo::operator!=(const ttv::UserInfo& other) const {
  return !(*this == other);
}

ttv::ChannelInfo::ChannelInfo()
    : channelId(0),
      createdAtTimestamp(0),
      updatedAtTimestamp(0),
      lastBroadcastEndTimestamp(0),
      numFollowers(0),
      numViews(0),
      mature(false),
      partner(false),
      affiliate(false) {}

ttv::StreamInfo::StreamInfo()
    : averageFPS(0.0),
      streamId(0),
      archiveVideoId(0),
      delay(0),
      viewerCount(0),
      videoHeight(0),
      createdAtTimestamp(0),
      broadcastPlatform(BroadcastPlatform::Unknown),
      streamType(StreamType::Unknown),
      isPlaylist(false) {}

ttv::StreamInfoUpdate::StreamInfoUpdate() : gameId(0) {}

ttv::WatchPartyUpdate::WatchPartyUpdate() : broadcastType(VodType::Unknown), viewable(false) {}

ttv::SquadMember::SquadMember() : channelId(0) {}

ttv::SquadInfo::SquadInfo() : ownerId(0), status(SquadStatus::Unknown) {}

ttv::TrackingValue::TrackingValue() : mType(Type::Null) {}

ttv::TrackingValue::TrackingValue(std::nullptr_t /*value*/) : mType(Type::Null) {}

ttv::TrackingValue::TrackingValue(bool value) : mType(Type::Boolean) {
  mValue.asBool = value;
}

ttv::TrackingValue::TrackingValue(int64_t value) : mType(Type::Integer) {
  mValue.asInteger = value;
}

ttv::TrackingValue::TrackingValue(uint32_t value) : mType(Type::Integer) {
  mValue.asInteger = value;
}

ttv::TrackingValue::TrackingValue(int value) : mType(Type::Integer) {
  mValue.asInteger = value;
}

ttv::TrackingValue::TrackingValue(double value) : mType(Type::Double) {
  mValue.asDouble = value;
}

ttv::TrackingValue::TrackingValue(const std::string& value) : mType(Type::String) {
  new (&mValue.asString) std::string{value};
}

ttv::TrackingValue::TrackingValue(std::string&& value) : mType(Type::String) {
  new (&mValue.asString) std::string{std::move(value)};
}

ttv::TrackingValue::TrackingValue(const char* value) : mType(Type::String) {
  new (&mValue.asString) std::string{std::move(value)};
}

ttv::TrackingValue::TrackingValue(const TrackingValue& src) : mType(Type::Null) {
  *this = src;
}

ttv::TrackingValue::TrackingValue(TrackingValue&& src) : mType(Type::Null) {
  *this = std::move(src);
}

ttv::TrackingValue::~TrackingValue() {
  if (mType == Type::String) {
    mValue.asString.~basic_string();
  }
}

ttv::TrackingValue& ttv::TrackingValue::operator=(const TrackingValue& src) {
  if (mType == Type::String) {
    mValue.asString.~basic_string();
  }

  switch (src.mType) {
    case Type::Null:
      break;
    case Type::Boolean:
      mValue.asBool = src.mValue.asBool;
      break;
    case Type::Integer:
      mValue.asInteger = src.mValue.asInteger;
      break;
    case Type::Double:
      mValue.asDouble = src.mValue.asDouble;
      break;
    case Type::String:
      new (&mValue.asString) std::string{src.mValue.asString};
      break;
  }
  mType = src.mType;
  return *this;
}

ttv::TrackingValue& ttv::TrackingValue::operator=(TrackingValue&& src) {
  if (mType == Type::String) {
    mValue.asString.~basic_string();
  }

  switch (src.mType) {
    case Type::Null:
      break;
    case Type::Boolean:
      mValue.asBool = src.mValue.asBool;
      break;
    case Type::Integer:
      mValue.asInteger = src.mValue.asInteger;
      break;
    case Type::Double:
      mValue.asDouble = src.mValue.asDouble;
      break;
    case Type::String:
      new (&mValue.asString) std::string{std::move(src.mValue.asString)};
      src.mValue.asString.~basic_string();
      break;
  }
  mType = src.mType;
  src.mType = Type::Null;

  return *this;
}

bool ttv::TrackingValue::GetBooleanValue() const {
  TTV_ASSERT(mType == Type::Boolean);
  return (mType == Type::Boolean) ? mValue.asBool : false;
}

int64_t ttv::TrackingValue::GetIntegerValue() const {
  TTV_ASSERT(mType == Type::Integer);
  return (mType == Type::Integer) ? mValue.asInteger : 0;
}

double ttv::TrackingValue::GetDoubleValue() const {
  TTV_ASSERT(mType == Type::Double);
  return (mType == Type::Double) ? mValue.asDouble : 0.0;
}

std::string ttv::TrackingValue::GetStringValue() const {
  TTV_ASSERT(mType == Type::String);
  return (mType == Type::String) ? mValue.asString : "";
}
