/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/user/user.h"

#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getusertask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/userlistener.h"
#include "twitchsdk/core/user/userrepository.h"

namespace {
const char* kLogger = "User";
}

ttv::User::User(UserId userId) : mOAuthToken(std::make_shared<OAuthToken>("")), mUserId(userId) {}

ttv::User::~User() {
  ttv::trace::Message(kLogger, MessageLevel::Debug, "User dtor");
}

std::string ttv::User::GetLoggerName() const {
  return kLogger;
}

void ttv::User::AddListener(const std::shared_ptr<IUserListener>& listener) {
  mListeners.AddListener(listener);
}

void ttv::User::RemoveListener(const std::shared_ptr<IUserListener>& listener) {
  mListeners.RemoveListener(listener);
}

void ttv::User::LogIn() {
  TTV_ASSERT(mUserId != 0);

  mListeners.Invoke(
    [this](const std::shared_ptr<IUserListener>& listener) { listener->OnUserLogInComplete(this, TTV_EC_SUCCESS); });
}

void ttv::User::LogOut() {
  TTV_ASSERT(mUserId != 0);

  mListeners.Invoke(
    [this](const std::shared_ptr<IUserListener>& listener) { listener->OnUserLogOutComplete(this, TTV_EC_SUCCESS); });
}

void ttv::User::ReportOAuthTokenInvalid(const std::shared_ptr<const OAuthToken>& oauthToken, TTV_ErrorCode ec) {
  TTV_ASSERT(mUserId != 0);

  // Invalid the current token
  if (oauthToken == mOAuthToken) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "ReportOAuthTokenInvalid issue [%s, %s]: %s",
      mUserInfo.userName.c_str(), oauthToken->GetToken().c_str(), ErrorToString(ec));

    mOAuthToken->SetValid(false);
  } else {
    ttv::trace::Message(kLogger, MessageLevel::Error,
      "ReportOAuthTokenInvalid issue on old token, ignoring [%s, %s]: %s", mUserInfo.userName.c_str(),
      oauthToken->GetToken().c_str(), ErrorToString(ec));
  }

  // Notify clients
  mListeners.Invoke([this, oauthToken, ec](const std::shared_ptr<IUserListener>& listener) {
    listener->OnUserAuthenticationIssue(this, oauthToken, ec);
  });
}

void ttv::User::SetUserInfo(const UserInfo& userInfo) {
  TTV_ASSERT(mUserId != 0);
  TTV_ASSERT(userInfo.userId == mUserId);
  mUserInfo = userInfo;
}

void ttv::User::SetOAuthToken(const std::shared_ptr<OAuthToken>& authToken) {
  TTV_ASSERT(mUserId != 0);

  mOAuthToken = authToken;
}

std::shared_ptr<const ttv::OAuthToken> ttv::User::GetOAuthToken() const {
  return std::const_pointer_cast<const OAuthToken>(mOAuthToken);
}

TTV_ErrorCode ttv::User::Initialize() {
  TTV_ErrorCode ec = Component::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    mComponentContainer = std::make_shared<ComponentContainer>();
    mComponentContainer->Initialize();
  }

  return ec;
}

void ttv::User::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  if (mComponentContainer != nullptr) {
    mComponentContainer->Update();
  }

  Component::Update();
}

TTV_ErrorCode ttv::User::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mComponentContainer != nullptr) {
      mComponentContainer->Shutdown();
    }
  }

  return ec;
}

bool ttv::User::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  if (mComponentContainer != nullptr && mComponentContainer->GetState() != ComponentContainer::State::Uninitialized) {
    return false;
  }

  return true;
}

void ttv::User::CompleteShutdown() {
  Component::CompleteShutdown();

  mOAuthToken.reset();
  mUserRepository.reset();
  mComponentContainer.reset();
}
