/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/user/userrepository.h"

#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getusertask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLoggerName = "UserRepository";
}

ttv::UserRepository::UserRepository() : mLookupInProgress(false) {}

TTV_ErrorCode ttv::UserRepository::Initialize() {
  TTV_ErrorCode ec = Component::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    mAnonymousUser = std::make_shared<User>(0);
    ec = mAnonymousUser->Initialize();
  }

  if (TTV_FAILED(ec)) {
    CompleteShutdown();
  }

  return ec;
}

std::string ttv::UserRepository::GetLoggerName() const {
  return kLoggerName;
}

void ttv::UserRepository::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  // Update current users
  mAnonymousUser->Update();

  for (auto iter = mUsers.begin(); iter != mUsers.end(); ++iter) {
    auto user = iter->second;
    user->Update();
  }

  // Cleanup unregistered users
  for (auto iter = mCleanupUsers.begin(); iter != mCleanupUsers.end();) {
    auto user = *iter;
    user->Update();

    if (user->GetState() == User::State::Uninitialized) {
      iter = mCleanupUsers.erase(iter);
    } else {
      ++iter;
    }
  }

  // Process the next user info lookup request
  if (!mLookupInProgress) {
    ScheduleUserInfoLookup();
  }

  Component::Update();
}

TTV_ErrorCode ttv::UserRepository::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    // Shutdown users
    if (mAnonymousUser != nullptr) {
      ec = mAnonymousUser->Shutdown();
      if (TTV_SUCCEEDED(ec)) {
        mCleanupUsers.push_back(mAnonymousUser);
      }
    }

    while (!mUsers.empty()) {
      std::shared_ptr<User> user = mUsers.begin()->second;
      ec = UnRegisterUser(user->GetUserId());
      TTV_ASSERT(TTV_SUCCEEDED(ec));
    }

    // Abort pending lookups
    for (auto iter = mPendingUserInfoLookups.begin(); iter != mPendingUserInfoLookups.end();) {
      if (iter->task == nullptr) {
        UserInfo userInfo;
        userInfo.userName = iter->username;

        ErrorDetails errorDetails(TTV_EC_SHUTTING_DOWN, "Shutting down");

        iter->callback(errorDetails, userInfo);
        iter = mPendingUserInfoLookups.erase(iter);
      } else {
        ++iter;
      }
    }
  }

  return ec;
}

bool ttv::UserRepository::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  if (mLookupInProgress) {
    return false;
  }

  if (!mCleanupUsers.empty()) {
    return false;
  }

  if (!mPendingUserInfoLookups.empty() && mPendingUserInfoLookups[0].task != nullptr) {
    return false;
  }

  return true;
}

void ttv::UserRepository::CompleteShutdown() {
  Component::CompleteShutdown();

  TTV_ASSERT(!mLookupInProgress);
  TTV_ASSERT(mCleanupUsers.empty());
  TTV_ASSERT(mUsers.empty());

  mCleanupUsers.clear();
  mUsers.clear();
  mUserInfo.clear();
  mPendingUserInfoLookups.clear();
  mAnonymousUser.reset();
}

std::shared_ptr<ttv::User> ttv::UserRepository::RegisterUser(UserId userId) {
  if (mState != State::Initialized || userId == 0) {
    return nullptr;
  }

  std::shared_ptr<User> user;

  // Lookup the user
  auto iter = mUsers.find(userId);
  if (iter != mUsers.end()) {
    user = iter->second;
  }

  // Create a new one
  if (user == nullptr) {
    user = std::make_shared<User>(userId);
    user->Initialize();

    mUsers[userId] = user;
  }

  return user;
}

std::shared_ptr<ttv::User> ttv::UserRepository::GetUser(UserId userId) {
  if (mState != State::Initialized || userId == 0) {
    return nullptr;
  }

  // Lookup the user
  auto iter = mUsers.find(userId);

  return (iter == mUsers.end()) ? nullptr : iter->second;
}

TTV_ErrorCode ttv::UserRepository::GetUsers(std::vector<std::shared_ptr<User>>& users) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUTTING_DOWN;
  }

  for (auto kvp : mUsers) {
    users.push_back(kvp.second);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::UserRepository::UnRegisterUser(UserId userId) {
  if (userId == 0) {
    return TTV_EC_INVALID_USERID;
  }

  auto iter = mUsers.find(userId);
  if (iter != mUsers.end()) {
    auto user = iter->second;

    user->LogOut();

    TTV_ErrorCode ec = user->Shutdown();
    if (TTV_SUCCEEDED(ec)) {
      mUsers.erase(iter);
      mCleanupUsers.push_back(user);
    }

    return ec;
  } else {
    return TTV_EC_INVALID_LOGIN;
  }
}

TTV_ErrorCode ttv::UserRepository::FetchUserInfoById(UserId userId, UserInfoLookupCallback callback) {
  if (userId == 0) {
    return TTV_EC_INVALID_USERID;
  } else if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  UserInfoLookupEntry entry;
  entry.userId = userId;
  entry.callback = callback;
  mPendingUserInfoLookups.push_back(entry);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::UserRepository::FetchUserInfoByName(const std::string& username, UserInfoLookupCallback callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  UserInfoLookupEntry entry;
  entry.username = username;
  entry.callback = callback;
  mPendingUserInfoLookups.push_back(entry);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::UserRepository::FetchUserInfoByAuthToken(
  const std::shared_ptr<OAuthToken>& authToken, UserInfoLookupCallback callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  UserInfoLookupEntry entry;
  entry.authToken = authToken;
  entry.callback = callback;
  mPendingUserInfoLookups.push_back(entry);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::UserRepository::GetUserInfoById(UserId userId, UserInfo& result) {
  if (userId == 0) {
    return TTV_EC_INVALID_USERID;
  }

  auto iter = mUserInfo.find(userId);
  if (iter == mUserInfo.end()) {
    return TTV_EC_USERINFO_NOT_AVAILABLE;
  }

  result = iter->second;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::UserRepository::GetUserInfoByName(const std::string& username, UserInfo& result) {
  auto lowercaseName = ToLowerCase(username);

  auto iter = std::find_if(mUserInfo.begin(), mUserInfo.end(),
    [&lowercaseName](const std::pair<UserId, UserInfo>& pair) { return lowercaseName == pair.second.userName; });

  if (iter == mUserInfo.end()) {
    return TTV_EC_USERINFO_NOT_AVAILABLE;
  }

  result = iter->second;

  return TTV_EC_SUCCESS;
}

void ttv::UserRepository::CompleteUserInfoLookup(const UserInfo& userInfo, const ErrorDetails& errorDetails) {
  mLookupInProgress = false;

  UserInfoLookupCallback callback = mPendingUserInfoLookups[0].callback;
  mPendingUserInfoLookups.erase(mPendingUserInfoLookups.begin());

  if (callback != nullptr) {
    callback(errorDetails, userInfo);
  }
}

TTV_ErrorCode ttv::UserRepository::ScheduleUserInfoLookup() {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (mLookupInProgress) {
    return TTV_EC_REQUEST_PENDING;
  }

  if (mPendingUserInfoLookups.empty()) {
    return TTV_EC_SUCCESS;
  }

  UserInfoLookupEntry& entry = mPendingUserInfoLookups[0];

  UserInfo currentInfo;
  currentInfo.userName = entry.username;
  currentInfo.userId = entry.userId;

  TTV_ErrorCode ec;

  GetUserTask::Callback fetchCallback = [this, currentInfo](GetUserTask* source, const ErrorDetails& errorDetails,
                                          std::shared_ptr<GetUserTask::Result> result) {
    CompleteTask(source);

    if (TTV_SUCCEEDED(errorDetails.ec)) {
      CacheUserInfo(result->userInfo);
      CompleteUserInfoLookup(result->userInfo, errorDetails);
    } else {
      CompleteUserInfoLookup(currentInfo, errorDetails);
    }
  };

  if (!entry.username.empty()) {
    entry.task = std::make_shared<GetUserTask>(entry.username, fetchCallback);
  } else if (entry.authToken != nullptr) {
    entry.task = std::make_shared<GetUserTask>(entry.authToken, fetchCallback);
  } else {
    entry.task = std::make_shared<GetUserTask>(entry.userId, fetchCallback);
  }
  ec = StartTask(entry.task);

  if (TTV_SUCCEEDED(ec)) {
    mLookupInProgress = true;
  } else {
    UserInfo userInfo;
    userInfo.userName = entry.username;
    userInfo.userId = entry.userId;

    ErrorDetails errorDetails(TTV_EC_SHUTTING_DOWN, "Task submission failed");
    CompleteUserInfoLookup(userInfo, errorDetails);
  }

  return ec;
}

TTV_ErrorCode ttv::UserRepository::CacheUserInfo(const UserInfo& userInfo) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!IsValidUserName(userInfo.userName) || userInfo.userId == 0) {
    return TTV_EC_INVALID_ARG;
  }

  mUserInfo[userInfo.userId] = userInfo;

  return TTV_EC_SUCCESS;
}
