/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "testwebsocket.h"
#include "testwebsocketfactory.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"

namespace ttv {
namespace test {
class PubSubTestUtility;
}
}  // namespace ttv

/**
 * A test utility for verifying the functionality of pub sub components.
 */
class ttv::test::PubSubTestUtility {
 public:
  PubSubTestUtility();

  void SetUpStubs();
  void SetUpStubs(const std::string& uri);
  void SetUpComponents(std::shared_ptr<ttv::TaskRunner> taskRunner, std::shared_ptr<ttv::User> user);
  void TearDownComponents();
  void TearDownStubs();

  std::shared_ptr<PubSubClient> GetPubSubClient() const { return mPubSubClient; }
  void SetListenErrorString(const std::string& errorString) { mListenErrorString = errorString; }

  /**
   * This class keeps track of the topics to which the client has subscribed. Returns true if the client has subscribed
   * to the passed-in topic.
   */
  bool IsSubscribedToTopic(const std::string& topic) const;

  /**
   * Push a pub-sub message to the client.
   * This method takes care of the top-level JSON that wraps the topic and the message data.
   */
  void PushPubSubMessage(const std::string& topic, const std::string& message);
  void PushPubSubMessage(const std::string& topic, const ttv::json::Value& message);
  void PushPubSubMessage(const std::string& topic, const char* message) {
    PushPubSubMessage(topic, std::string(message));
  }

 private:
  std::shared_ptr<TestWebSocketFactory> mTestWebSocketFactory;
  std::shared_ptr<TestWebSocket> mTestWebSocket;

  std::shared_ptr<PubSubClient> mPubSubClient;
  std::vector<std::string> mSubscribedTopics;
  std::unique_ptr<IMutex> mMutex;

  std::string mListenErrorString;

  void ProcessSentPayload(const std::string& sentPayload);
};
