/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/channel/ichannellistener.h"
#include "twitchsdk/core/coreapi.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class TestChannelListener : public IChannelListener {
 public:
  using StreamUpCallback = std::function<void(uint32_t playDelaySeconds)>;
  using StreamDownCallback = std::function<void()>;
  using StreamViewerCountChangedCallback = std::function<void(uint32_t viewerCount)>;
  using StreamTriggeredMidrollCallback = std::function<void(uint32_t durationSeconds)>;
  using StreamReceivedWatchPartyUpdateCallback = std::function<void(const WatchPartyUpdate& update)>;
  using ProfileImageUpdatedCallback = std::function<void(const std::vector<ProfileImage>& images)>;
  using StreamInfoUpdatedCallback = std::function<void(StreamInfoUpdate&& info)>;
  using SquadUpdatedCallback = std::function<void(SquadInfo&& info)>;
  using SquadLeftCallback = std::function<void()>;
  using PixelTrackingUpdateCallback = std::function<void(bool refresh)>;

  virtual void StreamUp(uint32_t playDelaySeconds) override {
    if (streamUpCallback != nullptr) {
      streamUpCallback(playDelaySeconds);
    }
  }

  virtual void StreamDown() override {
    if (streamDownCallback != nullptr) {
      streamDownCallback();
    }
  }

  virtual void StreamViewerCountChanged(uint32_t viewerCount) override {
    if (streamViewerCountChangedCallback != nullptr) {
      streamViewerCountChangedCallback(viewerCount);
    }
  }

  virtual void StreamTriggeredMidroll(uint32_t durationSeconds) override {
    if (streamTriggeredMidrollCallback != nullptr) {
      streamTriggeredMidrollCallback(durationSeconds);
    }
  }

  virtual void StreamReceivedWatchPartyUpdate(const WatchPartyUpdate& update) override {
    if (streamReceivedWatchPartyUpdateCallback != nullptr) {
      streamReceivedWatchPartyUpdateCallback(update);
    }
  }

  virtual void ProfileImageUpdated(const std::vector<ProfileImage>& images) override {
    if (profileImageUpdatedCallback != nullptr) {
      profileImageUpdatedCallback(images);
    }
  }

  virtual void StreamInfoUpdated(StreamInfoUpdate&& info) override {
    if (streamInfoUpdatedCallback != nullptr) {
      streamInfoUpdatedCallback(std::move(info));
    }
  }

  virtual void SquadUpdated(SquadInfo&& info) override {
    if (squadUpdatedCallback != nullptr) {
      squadUpdatedCallback(std::move(info));
    }
  }

  virtual void SquadLeft() override {
    if (squadLeftCallback != nullptr) {
      squadLeftCallback();
    }
  }

  virtual void PixelTrackingUpdate(bool refresh) override {
    if (pixelTrackingUpdateCallback != nullptr) {
      pixelTrackingUpdateCallback(refresh);
    }
  }

  StreamUpCallback streamUpCallback;
  StreamDownCallback streamDownCallback;
  StreamViewerCountChangedCallback streamViewerCountChangedCallback;
  StreamTriggeredMidrollCallback streamTriggeredMidrollCallback;
  StreamReceivedWatchPartyUpdateCallback streamReceivedWatchPartyUpdateCallback;
  ProfileImageUpdatedCallback profileImageUpdatedCallback;
  StreamInfoUpdatedCallback streamInfoUpdatedCallback;
  SquadUpdatedCallback squadUpdatedCallback;
  SquadLeftCallback squadLeftCallback;
  PixelTrackingUpdateCallback pixelTrackingUpdateCallback;
};

class ChannelListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    InitializeModule(mCoreApi);
    AddModule(mCoreApi);

    mUserInfo.userId = 9001;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mChannelListener = std::make_shared<TestChannelListener>();
  }

  std::shared_ptr<CoreAPI> mCoreApi;

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestChannelListener> mChannelListener;
};
}  // namespace

TEST_F(ChannelListenerTest, AnonymousReconnect) {
  bool receivedCallback = false;
  mChannelListener->streamUpCallback = [&receivedCallback](uint32_t playDelaySeconds) {
    EXPECT_EQ(playDelaySeconds, 30);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(0, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
  receivedCallback = false;

  mCoreApi->DisconnectPubSub(0);

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_FALSE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  mCoreApi->ConnectPubSub(0);

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(5000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, LoggedInReconnect) {
  bool receivedCallback = false;
  mChannelListener->streamUpCallback = [&receivedCallback](uint32_t playDelaySeconds) {
    EXPECT_EQ(playDelaySeconds, 30);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
  receivedCallback = false;

  mCoreApi->DisconnectPubSub(mUserInfo.userId);

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_FALSE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  mCoreApi->ConnectPubSub(mUserInfo.userId);

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(5000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamUp) {
  bool receivedCallback = false;
  mChannelListener->streamUpCallback = [&receivedCallback](uint32_t playDelaySeconds) {
    EXPECT_EQ(playDelaySeconds, 30);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamDown) {
  bool receivedCallback = false;
  mChannelListener->streamDownCallback = [&receivedCallback]() { receivedCallback = true; };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-down\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamViewerCountChanged) {
  bool receivedCallback = false;
  mChannelListener->streamViewerCountChangedCallback = [&receivedCallback](uint32_t viewerCount) {
    EXPECT_EQ(viewerCount, 1337);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"viewcount\", \"viewers\":1337}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamTriggeredMidroll) {
  bool receivedCallback = false;
  mChannelListener->streamTriggeredMidrollCallback = [&receivedCallback](uint32_t durationSeconds) {
    EXPECT_EQ(durationSeconds, 15);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"commercial\", \"length\":15}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, ProfileImageUpdated) {
  bool receivedCallback = false;
  mChannelListener->profileImageUpdatedCallback = [&receivedCallback](const std::vector<ProfileImage>& images) {
    ASSERT_EQ(images.size(), 2);
    ASSERT_EQ(images[0].width, 150);
    ASSERT_EQ(images[0].height, 150);
    ASSERT_EQ(images[0].url, "https://150x150.png");
    ASSERT_EQ(images[0].format, "png");
    ASSERT_EQ(images[1].width, 28);
    ASSERT_EQ(images[1].height, 28);
    ASSERT_EQ(images[1].url, "https://28x28.png");
    ASSERT_EQ(images[1].format, "png");
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("user-image-update.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("user-image-update.12345",
    "{\"user_id\":\"12345\",\"type\":\"user_image_update\",\"image_type\":\"profile_image\",\"new_image\":{\"150x150\":{\"width\":150,\"height\":150,\"url\":\"https://150x150.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"150x150\",\"format\":\"png\",\"new_format\":true},\"28x28\":{\"width\":28,\"height\":28,\"url\":\"https://28x28.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"28x28\",\"format\":\"png\",\"new_format\":true}},\"status\":\"SUCCESS\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamReceivedWatchPartyUpdate) {
  bool receivedCallback = false;
  mChannelListener->streamReceivedWatchPartyUpdateCallback = [&receivedCallback](const WatchPartyUpdate& update) {
    EXPECT_EQ(update.incrementUrl, "http://url.com");
    EXPECT_EQ(update.vodId, "v12345");
    EXPECT_EQ(update.watchPartyId, "watchPartyId12345");
    EXPECT_EQ(update.broadcastType, VodType::Archive);
    EXPECT_TRUE(update.viewable);
    EXPECT_EQ(update.title, "My Stream Title");

    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345",
    "{"
    "\"type\":\"watchparty-vod\","
    "\"vod\":{"
    "\"increment_url\":\"http://url.com\","
    "\"vod_id\":\"v12345\","
    "\"title\":\"My Stream Title\","
    "\"wp_id\":\"watchPartyId12345\","
    "\"broadcast_type\":\"archive\","
    "\"viewable\":\"public\""
    "}"
    "}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, StreamInfoUpdated) {
  bool receivedCallback = false;
  mChannelListener->streamInfoUpdatedCallback = [&receivedCallback](StreamInfoUpdate&& info) {
    ASSERT_EQ(info.title, "stream title");
    ASSERT_EQ(info.game, "Transistor");
    ASSERT_EQ(info.gameId, 1000);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("broadcast-settings-update.12345"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("broadcast-settings-update.12345",
    "{\"channel_id\":\"12345\",\"channel\":\"channelName\",\"status\":\"stream title\",\"game\":\"Transistor\",\"game_id\":\"1000\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, MultipleListeners) {
  uint8_t numCallbacks = 0;
  uint8_t numStreamUp = 0;
  uint8_t numStreamDown = 0;

  mChannelListener->streamUpCallback = [&numCallbacks, &numStreamUp](uint32_t playDelaySeconds) {
    EXPECT_EQ(playDelaySeconds, 30);
    numCallbacks++;
    numStreamUp++;
  };

  mChannelListener->streamDownCallback = [&numCallbacks, &numStreamDown]() {
    numCallbacks++;
    numStreamDown++;
  };

  std::shared_ptr<IChannelStatus> channelStatus1;
  std::shared_ptr<IChannelStatus> channelStatus2;

  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus1);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 54321, mChannelListener, channelStatus2);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() {
      return mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.12345") &&
             mPubSubTestUtility.IsSubscribedToTopic("video-playback-by-id.54321");
    },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.54321", "{\"type\":\"stream-up\", \"play_delay\":30}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-down\"}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-up\", \"play_delay\":30}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.54321", "{\"type\":\"stream-down\"}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.12345", "{\"type\":\"stream-down\"}");
  mPubSubTestUtility.PushPubSubMessage("video-playback-by-id.54321", "{\"type\":\"stream-up\", \"play_delay\":30}");

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [&numCallbacks, &numStreamUp, &numStreamDown]() {
      return numCallbacks == 7 && numStreamUp == 4 && numStreamDown == 3;
    },
    GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, PixelTrackingUpdateTrue) {
  bool receivedCallback = false;
  mChannelListener->pixelTrackingUpdateCallback = [&receivedCallback](bool refresh) {
    EXPECT_TRUE(refresh);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("ad-property-refresh.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage(
    "ad-property-refresh.12345", "{\"type\":\"pixel_tracking_update\", \"data\":{\"refresh\":\"true\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChannelListenerTest, PixelTrackingUpdateFalse) {
  bool receivedCallback = false;
  mChannelListener->pixelTrackingUpdateCallback = [&receivedCallback](bool refresh) {
    EXPECT_FALSE(refresh);
    receivedCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(mUserInfo.userId, 12345, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("ad-property-refresh.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage(
    "ad-property-refresh.12345", "{\"type\":\"pixel_tracking_update\", \"data\":{\"refresh\":\"false\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}