/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/
#include "fixtures/repositorytest.h"
#include "testsystemclock.h"
#include "testutilities.h"
#include "twitchsdk/core/channel/channelrepository.h"

#include <stdio.h>

using namespace ttv;

TEST_F(RepositoryTest, SimpleLookup) {
  std::shared_ptr<ChannelRepository> channelRepository = std::make_shared<ChannelRepository>();
  AddComponent(channelRepository);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "12826")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/channel_twitch.json")
    .Done();

  channelRepository->FetchChannelInfo(12826, nullptr);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> hasChannel = [&channelRepository]() {
    ChannelInfo result;
    TTV_ErrorCode ec = channelRepository->GetChannelInfo(12826, result);
    return TTV_SUCCEEDED(ec);
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, hasChannel, updateFunction));
}

TEST_F(RepositoryTest, NoResultFetchAndGet) {
  std::shared_ptr<ChannelRepository> channelRepository = std::make_shared<ChannelRepository>();
  AddComponent(channelRepository);
  channelRepository->FetchChannelInfo(1001, nullptr);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> hasChannel = [&channelRepository]() {
    ChannelInfo result;
    TTV_ErrorCode ec = channelRepository->GetChannelInfo(1001, result);
    return TTV_FAILED(ec);
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, hasChannel, updateFunction));
}

TEST_F(RepositoryTest, MultipleLookupsInOrder) {
  std::shared_ptr<ChannelRepository> channelRepository = std::make_shared<ChannelRepository>();
  AddComponent(channelRepository);
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "12826")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/channel_twitch.json");

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "12827")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/channel_twitch_2.json");

  channelRepository->FetchChannelInfo(12826, nullptr);
  channelRepository->FetchChannelInfo(12827, nullptr);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> onlyResultsFromFirstLookup = [&channelRepository]() {
    ChannelInfo result1;
    TTV_ErrorCode ec1 = channelRepository->GetChannelInfo(12826, result1);
    ChannelInfo result2;
    TTV_ErrorCode ec2 = channelRepository->GetChannelInfo(12827, result2);
    return (TTV_SUCCEEDED(ec1) && TTV_FAILED(ec2));
  };

  std::function<bool()> haveResultsFromBothLookups = [&channelRepository]() {
    ChannelInfo result1;
    TTV_ErrorCode ec1 = channelRepository->GetChannelInfo(12826, result1);
    ChannelInfo result2;
    TTV_ErrorCode ec2 = channelRepository->GetChannelInfo(12827, result2);
    return (TTV_SUCCEEDED(ec1) && TTV_SUCCEEDED(ec2));
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, onlyResultsFromFirstLookup, updateFunction));
  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, haveResultsFromBothLookups, updateFunction));
}

TEST_F(RepositoryTest, ForceRefresh) {
  std::shared_ptr<ChannelRepository> channelRepository = std::make_shared<ChannelRepository>();
  AddComponent(channelRepository);
  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .AddJsonValue({"variables", "userID"}, "12826")
                             .SetStatusCode(200)
                             .SetResponseBodyFromFile("core/channel_twitch.json");

  channelRepository->FetchChannelInfo(12826, nullptr);
  channelRepository->FetchChannelInfo(12826, nullptr);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> requestedTwice = [&response]() { return response.GetRequestCount() == 2; };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, requestedTwice, updateFunction));
}

TEST_F(RepositoryTest, PurgeUnused) {
  std::shared_ptr<ChannelRepository> channelRepository = std::make_shared<ChannelRepository>();
  AddComponent(channelRepository);
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "12826")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/channel_twitch.json");

  channelRepository->FetchChannelInfo(12826, nullptr);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> gotResult = [&channelRepository]() {
    ChannelInfo result;
    TTV_ErrorCode ec = channelRepository->GetChannelInfo(12826, result);
    return TTV_SUCCEEDED(ec);
  };

  std::function<bool()> noResult = [&channelRepository]() {
    ChannelInfo result;
    TTV_ErrorCode ec = channelRepository->GetChannelInfo(12826, result);
    return TTV_FAILED(ec);
  };

  ttv::test::WaitUntilResultWithPollTask(1000, gotResult, updateFunction);
  mTestSystemClock->SetOffset(60 * 60 * 1000);
  Update();

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, noResult, updateFunction));

  mTestSystemClock->SetOffset(0);
}
