/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class ApiClientProxy : public ICoreApiClient {
 public:
  using GetClientNameFunc = std::function<std::string()>;
  using GetRequiredOAuthScopesFunc = std::function<void(std::vector<std::string>&)>;
  using CoreUserLoggedInFunc = std::function<void(std::shared_ptr<User>)>;
  using CoreUserLoggedOutFunc = std::function<void(std::shared_ptr<User>)>;

  GetClientNameFunc mGetClientName;
  GetRequiredOAuthScopesFunc mGetRequiredOAuthScopes;
  CoreUserLoggedInFunc mCoreUserLoggedIn;
  CoreUserLoggedOutFunc mCoreUserLoggedOut;

  virtual std::string GetClientName() override {
    if (mGetClientName != nullptr) {
      return mGetClientName();
    }
    return "TestClient";
  }

  virtual void GetRequiredOAuthScopes(std::vector<std::string>& scopes) override {
    if (mGetRequiredOAuthScopes != nullptr) {
      mGetRequiredOAuthScopes(scopes);
    }
  }

  virtual void CoreUserLoggedIn(std::shared_ptr<User> user) override {
    if (mCoreUserLoggedIn != nullptr) {
      mCoreUserLoggedIn(user);
    }
  }

  virtual void CoreUserLoggedOut(std::shared_ptr<User> user) override {
    if (mCoreUserLoggedOut != nullptr) {
      mCoreUserLoggedOut(user);
    }
  }
};
}  // namespace

TEST_F(SdkBaseTest, TestLoginLogout) {
  auto coreAPI = std::make_shared<CoreAPI>();
  AddModule(coreAPI);

  auto userInfo = mUserTestUtility.DefaultUserInfo();
  std::string authToken = "auth_token";
  mUserTestUtility.RegisterUserInfo(authToken, userInfo);

  InitializeModule(coreAPI);

  auto fakeClient = std::make_shared<ApiClientProxy>();
  ASSERT_TRUE(TTV_SUCCEEDED(coreAPI->RegisterClient(fakeClient)));

  bool loggedIn = false;
  fakeClient->mCoreUserLoggedIn = [&loggedIn, &userInfo](std::shared_ptr<User> user) {
    loggedIn = true;

    EXPECT_EQ(user->GetUserId(), userInfo.userId);
    EXPECT_EQ(user->GetUserName(), userInfo.userName);
  };

  bool loggedOut = false;
  fakeClient->mCoreUserLoggedOut = [&loggedOut, &userInfo](std::shared_ptr<User> user) {
    loggedOut = true;

    EXPECT_EQ(user->GetUserId(), userInfo.userId);
    EXPECT_EQ(user->GetUserName(), userInfo.userName);
  };

  EXPECT_TRUE(TTV_SUCCEEDED(LogIn(coreAPI, authToken, userInfo)));

  EXPECT_TRUE(WaitUntilResult(1000, [&loggedIn]() { return loggedIn; }));

  EXPECT_FALSE(loggedOut);

  loggedIn = false;

  EXPECT_TRUE(TTV_SUCCEEDED(LogOut(coreAPI, userInfo)));

  EXPECT_TRUE(WaitUntilResult(1000, [&loggedOut]() { return loggedOut; }));

  EXPECT_FALSE(loggedIn);

  EXPECT_TRUE(TTV_SUCCEEDED(coreAPI->UnregisterClient(fakeClient)));
}
