/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "fixtures/tasktest.h"
#include "testutilities.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getchanneltask.h"
#include "twitchsdk/core/task/getprofileimageurltask.h"
#include "twitchsdk/core/task/getusertask.h"
#include "twitchsdk/core/task/simplejsonhttptask.h"
#include "twitchsdk/core/task/validateoauthtask.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;

namespace {
static const UserId kTestUserId = 9001;
static const std::string kTestOAuthToken = "***********";
static const HttpRequestType kTestHttpRequestType = HTTP_GET_REQUEST;
}  // namespace

TEST_F(TaskTest, GetUserTask_Success) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetStatusCode(200)
                    .SetResponseBodyFromFile("core/getuserbyid.json")
                    .Done();

  TestSimpleRequest<GetUserTask, ErrorDetails>(kTestUserId, [](GetUserTask* /*source*/,
                                                              const ErrorDetails& errorDetails,
                                                              std::shared_ptr<GetUserTask::Result> result) {
    ASSERT_TRUE(TTV_SUCCEEDED(errorDetails.ec));
    ASSERT_TRUE(result != nullptr);

    UserInfo& userInfo = result->userInfo;
    ASSERT_EQ(userInfo.userId, 12826);
    ASSERT_EQ(userInfo.userName, "twitch");
    ASSERT_EQ(userInfo.displayName, "Twitch");
    ASSERT_EQ(userInfo.logoImageUrl,
      "https://static-cdn.jtvnw.net/jtv_user_pictures/twitch-profile_image-8a8c5be2e3b64a9a-300x300.png");
    ASSERT_EQ(userInfo.bio,
      "Twitch is the world's leading video platform and community for gamers with more than 100 million visitors per month. Our mission is to connect gamers around the world by allowing them to broadcast, watch, and chat from everywhere they play.");

    Timestamp expectedCreatedTimestamp;
    ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2007-05-22T10:39:54Z", expectedCreatedTimestamp));
    ASSERT_EQ(userInfo.createdTimestamp, expectedCreatedTimestamp);
  });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetUserTask_NotFound) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetStatusCode(404)
                    .Done();

  TestSimpleRequest<GetUserTask, ErrorDetails>(kTestUserId,
    [](GetUserTask* /*source*/, const ErrorDetails& errorDetails, std::shared_ptr<GetUserTask::Result> /*result*/) {
      ASSERT_EQ(errorDetails.ec, TTV_EC_API_REQUEST_FAILED);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetUserTask_AuthenticationFailure) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetStatusCode(401)
                    .Done();

  TestSimpleRequest<GetUserTask, ErrorDetails>(kTestUserId,
    [](GetUserTask* /*source*/, const ErrorDetails& errorDetails, std::shared_ptr<GetUserTask::Result> /*result*/) {
      ASSERT_EQ(errorDetails.ec, TTV_EC_AUTHENTICATION);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetUserTask_ParseFailure) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetResponseBody("This is not parseable JSON")
                    .Done();

  TestSimpleRequest<GetUserTask, ErrorDetails>(kTestUserId,
    [](GetUserTask* /*source*/, const ErrorDetails& errorDetails, std::shared_ptr<GetUserTask::Result> /*result*/) {
      ASSERT_EQ(errorDetails.ec, TTV_EC_WEBAPI_RESULT_INVALID_JSON);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetChannelTask_Success) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetResponseBodyFromFile("core/getchanneltask.json")
                    .Done();

  TestSimpleRequest<GetChannelTask>(
    kTestUserId, [](GetChannelTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<GetChannelTask::Result> result) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_TRUE(result != nullptr);

      ChannelInfo channelInfo = result->channelInfo;

      ASSERT_EQ(channelInfo.displayName, "Twitch");
      ASSERT_EQ(channelInfo.name, "twitch");
      ASSERT_EQ(channelInfo.game, "E3 2016");
      ASSERT_EQ(channelInfo.description, "Test description");
      ASSERT_EQ(channelInfo.status, "#E3onTwitch - Coverage begins June 12th!");
      ASSERT_EQ(channelInfo.language, "EN");
      ASSERT_EQ(channelInfo.broadcasterLanguage, "EN");
      ASSERT_EQ(channelInfo.logoImageUrl,
        "https://static-cdn.jtvnw.net/jtv_user_pictures/twitch-profile_image-42277a83919bd975-300x300.png");
      ASSERT_EQ(channelInfo.channelUrl, "https://www.twitch.tv/twitch");
      ASSERT_EQ(channelInfo.videoBannerImageUrl,
        "https://static-cdn.jtvnw.net/jtv_user_pictures/twitch-channel_offline_image-9dc5c2bc4d9303e2-1920x1080.png");
      ASSERT_EQ(channelInfo.profileBannerImageUrl,
        "https://static-cdn.jtvnw.net/jtv_user_pictures/twitch-profile_banner-6936c61353e4aeed-480.png");

      ASSERT_EQ(channelInfo.channelId, 12826);

      Timestamp expectedCreatedTimestamp;
      ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2007-05-22T10:39:54Z", expectedCreatedTimestamp));
      ASSERT_EQ(channelInfo.createdAtTimestamp, expectedCreatedTimestamp);

      Timestamp expectedUpdatedTimestamp;
      ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2016-06-09T01:03:40Z", expectedUpdatedTimestamp));
      ASSERT_EQ(channelInfo.updatedAtTimestamp, expectedUpdatedTimestamp);

      ASSERT_EQ(channelInfo.numFollowers, 429143);
      ASSERT_EQ(channelInfo.numViews, 79011037);

      ASSERT_FALSE(channelInfo.mature);
      ASSERT_TRUE(channelInfo.partner);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetChannelTask_NotFound) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetStatusCode(404)
                    .Done();

  TestSimpleRequest<GetChannelTask>(
    kTestUserId, [](GetChannelTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<GetChannelTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_API_REQUEST_FAILED);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetChannelTask_AuthenticationFailure) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetStatusCode(401)
                    .Done();

  TestSimpleRequest<GetChannelTask>(
    kTestUserId, [](GetChannelTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<GetChannelTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_AUTHENTICATION);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetChannelTask_ParseFailure) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetType(HTTP_POST_REQUEST)
                    .AddJsonValue({"variables", "userID"}, ttv::json::Value(std::to_string(kTestUserId)))
                    .SetResponseBody("This is not parseable JSON")
                    .Done();

  TestSimpleRequest<GetChannelTask>(
    kTestUserId, [](GetChannelTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<GetChannelTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_WEBAPI_RESULT_INVALID_JSON);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, ValidateOAuth_Success) {
  auto response = mHttpRequest->AddResponse("https://api.twitch.tv/kraken?oauth_token=" + kTestOAuthToken)
                    .SetResponseBodyFromFile("core/validate_oauth_valid.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<ValidateOAuthTask> task;

  ValidateOAuthTask::Callback callback = [&callbackCalled, &task](ValidateOAuthTask* source, TTV_ErrorCode ec,
                                           std::shared_ptr<ValidateOAuthTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_TRUE(result->valid == true);
    ASSERT_TRUE(result->userName == "Twitch");
    ASSERT_TRUE(result->scopes.size() == 25);
    ASSERT_TRUE(result->scopes[0] == "channel_check_subscription");
    ASSERT_TRUE(result->scopes[1] == "channel_commercial");
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<ValidateOAuthTask>(kTestOAuthToken, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, ValidateOAuth_Failure) {
  auto response = mHttpRequest->AddResponse("https://api.twitch.tv/kraken?oauth_token=" + kTestOAuthToken)
                    .SetResponseBodyFromFile("core/validate_oauth_invalid.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<ValidateOAuthTask> task;

  ValidateOAuthTask::Callback callback = [&callbackCalled, &task](ValidateOAuthTask* source, TTV_ErrorCode ec,
                                           std::shared_ptr<ValidateOAuthTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_TRUE(result->valid == false);
    ASSERT_TRUE(result->userName.empty());
    ASSERT_TRUE(result->scopes.empty());
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<ValidateOAuthTask>(kTestOAuthToken, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, GetProfileImageUrlTask_Success) {
  auto response =
    mHttpRequest->AddResponse("https://api.twitch.tv/kraken/users/" + std::to_string(kTestUserId) + "/upload_image")
      .SetResponseBodyFromFile("core/getprofileimageurltask.json")
      .SetType(ttv::HttpRequestType::HTTP_POST_REQUEST)
      .Done();

  bool callbackCalled = false;
  std::shared_ptr<GetProfileImageUrlTask> task;

  GetProfileImageUrlTask::Callback callback = [&callbackCalled, &task](GetProfileImageUrlTask* source, TTV_ErrorCode ec,
                                                std::shared_ptr<GetProfileImageUrlTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_TRUE(result->uploadId == "f61b6216-f2a6-45ca-a8d5-8f5613d88093");
    ASSERT_TRUE(
      result->uploadUrl ==
      "https://prod-web-upload-service-ingest.s3-accelerate.amazonaws.com/f61b6216-f2a6-45ca-a8d5-8f5613d88093?X-Amz-Algorithm=AWS4-HMAC-SHA256\u0026X-Amz-Credential=ASIAJ4MSIQNVTARCK7HQ%2F20170911%2Fus-west-2%2Fs3%2Faws4_request\u0026X-Amz-Date=20170911T200456Z\u0026X-Amz-Expires=900\u0026X-Amz-Security-Token=FQoDYXdzENz%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDP91q2ARZTdu%2FLqL3CK3A1t7gyqQgBtoTLlRYVkjS63oowXyFcsUVdc9k%2BbD7fjLSYBzlL1rT1yOCdtkf6ICvzdFx94ptLUDm1EYdb2v5gS9m23O%2B%2BzrR7JWMJ6r2ikWkmSc2%2BWn52nV7XXB2nfE3pd4KaUiuTZnooH7pjWY0i%2Btygruejzs5Qe4YyVwo9fhP8rytI%2FTaeCACW8gtWZtyc3ENIEP9xjXReMgoEb%2B%2BISegP42M%2FasdGN5aIpN5mbO31K1lkOef4kin8y0x3tYzDOkLkblwRkIUiUXEM%2FbN3PLsY94ndqgz1%2FkANZ81cfXCSL%2BDngne3tXBAyNrZGN%2Fmgdtt5dv2hfROgRfesI2lViJ3Ccr9N%2FDi%2FDHw8GEna%2FJUqU%2B%2B%2FxF%2B%2FyQtNPfjt4cy2l1a9Yowt6Z2b7lhFJtsUK5AmN4GEaSacBJscTSeB4%2B5hmH2AgCYiWnlhFfxzb33tpFxdic2Jn%2F2IzTE6UCl6cf7rR9uHkQXPMPMBEMqmLMyaGH7gIQ1B9vPicPeyq%2BjRyb2gS6dobjIIoHH8M%2BBYxO1b01yPAkEga3k0DaAE%2FLnxu8RIX5mUTQLwICKYNQi0R7Zivccoonr%2FbzQU%3D\u0026X-Amz-SignedHeaders=host\u0026X-Amz-Signature=c0db691066213655041616f67276e405cd77eb9f659aeddfecbefbad54e5205f");
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<GetProfileImageUrlTask>(kTestUserId, kTestOAuthToken, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SimpleJsonHttpTask_Success) {
  const std::string kTestUrl = std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId);

  auto response = mHttpRequest->AddResponse(kTestUrl).SetResponseBodyFromFile("core/simplejsonhttptask.json").Done();

  TestSimpleJsonHttpRequest(kTestUrl, kTestHttpRequestType, kTestOAuthToken,
    [](SimpleJsonHttpTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<SimpleJsonHttpTask::Result> result) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_TRUE(result != nullptr);

      json::Value jsonValue = result->json;
      ASSERT_EQ(jsonValue["display_name"], "Twitch");
      ASSERT_EQ(jsonValue["_id"].asInt(), 12826);
      ASSERT_EQ(jsonValue["name"], "twitch");
      ASSERT_EQ(jsonValue["type"], "user");
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, SimpleJsonHttpTask_NotFound) {
  const std::string kTestUrl = std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId);

  auto response = mHttpRequest->AddResponse(kTestUrl).SetStatusCode(404).Done();

  TestSimpleJsonHttpRequest(kTestUrl, kTestHttpRequestType, kTestOAuthToken,
    [](SimpleJsonHttpTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<SimpleJsonHttpTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_API_REQUEST_FAILED);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, SimpleJsonHttpTask_AuthenticationFailure) {
  const std::string kTestUrl = std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId);

  auto response = mHttpRequest->AddResponse(kTestUrl).SetStatusCode(401).Done();

  TestSimpleJsonHttpRequest(kTestUrl, kTestHttpRequestType, kTestOAuthToken,
    [](SimpleJsonHttpTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<SimpleJsonHttpTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_AUTHENTICATION);
    });

  response->AssertRequestsMade();
}

TEST_F(TaskTest, SimpleJsonHttpTask_ParseFailure) {
  const std::string kTestUrl = std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId);

  auto response = mHttpRequest->AddResponse(kTestUrl).SetResponseBody("This is not parseable JSON").Done();

  TestSimpleJsonHttpRequest(kTestUrl, kTestHttpRequestType, kTestOAuthToken,
    [](SimpleJsonHttpTask* /*source*/, TTV_ErrorCode ec, std::shared_ptr<SimpleJsonHttpTask::Result> /*result*/) {
      ASSERT_EQ(ec, TTV_EC_WEBAPI_RESULT_INVALID_JSON);
    });

  response->AssertRequestsMade();
}
