/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/idashboardactivitylistener.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class TestDashboardActivityListener : public IDashboardActivityListener {
 public:
  using EventBitsUseageCallback = std::function<void(const DashboardActivityBitsUsage)>;
  using EventFollowerCallback = std::function<void(const DashboardActivityFollow)>;
  using EventAutoHostStartCallback = std::function<void(const DashboardActivityHost)>;
  using EventHostStartCallback = std::function<void(const DashboardActivityHost)>;
  using EventRaidingCallback = std::function<void(const DashboardActivityRaiding)>;
  using EventSubscriptionCallback = std::function<void(const DashboardActivitySubscription)>;
  using EventPrimeSubscriptionCallback = std::function<void(const DashboardActivitySubscription)>;
  using EventResubscriptionSharingCallback = std::function<void(const DashboardActivityResubscriptionSharing)>;
  using EventPrimeResubscriptionSharingCallback = std::function<void(const DashboardActivityResubscriptionSharing)>;
  using EventSubscriptionGiftingCommunityCallback =
    std::function<void(const DashboardActivitySubscriptionGiftingCommunity)>;
  using EventSubscriptionGiftingIndividualCallback =
    std::function<void(const DashboardActivitySubscriptionGiftingIndividual)>;
  using EventStreamUpCallback = std::function<void(const DashboardActivityHeader)>;

  ~TestDashboardActivityListener() override = default;

  void EventBitsUseage(const DashboardActivityBitsUsage& data) override {
    if (mEventBitsUseageCallback != nullptr) {
      mEventBitsUseageCallback(data);
    }
  }

  void EventFollower(const DashboardActivityFollow& data) override {
    if (mEventFollowerCallback != nullptr) {
      mEventFollowerCallback(data);
    }
  }

  void EventAutoHostStart(const DashboardActivityHost& data) override {
    if (mEventAutoHostStartCallback != nullptr) {
      mEventAutoHostStartCallback(data);
    }
  }

  void EventHostStart(const DashboardActivityHost& data) override {
    if (mEventHostStartCallback != nullptr) {
      mEventHostStartCallback(data);
    }
  }

  void EventRaiding(const DashboardActivityRaiding& data) override {
    if (mEventRaidingCallback != nullptr) {
      mEventRaidingCallback(data);
    }
  }

  void EventSubscription(const DashboardActivitySubscription& data) override {
    if (mEventSubscriptionCallback != nullptr) {
      mEventSubscriptionCallback(data);
    }
  }

  void EventPrimeSubscription(const DashboardActivitySubscription& data) override {
    if (mEventPrimeSubscriptionCallback != nullptr) {
      mEventPrimeSubscriptionCallback(data);
    }
  }

  void EventResubscriptionSharing(const DashboardActivityResubscriptionSharing& data) override {
    if (mEventResubscriptionSharingCallback != nullptr) {
      mEventResubscriptionSharingCallback(data);
    }
  }

  void EventPrimeResubscriptionSharing(const DashboardActivityResubscriptionSharing& data) override {
    if (mEventPrimeResubscriptionSharingCallback != nullptr) {
      mEventPrimeResubscriptionSharingCallback(data);
    }
  }

  void EventSubscriptionGiftingCommunity(const DashboardActivitySubscriptionGiftingCommunity& data) override {
    if (mEventSubscriptionGiftingCommunityCallback != nullptr) {
      mEventSubscriptionGiftingCommunityCallback(data);
    }
  }

  void EventSubscriptionGiftingIndividual(const DashboardActivitySubscriptionGiftingIndividual& data) override {
    if (mEventSubscriptionGiftingIndividualCallback != nullptr) {
      mEventSubscriptionGiftingIndividualCallback(data);
    }
  }

  void EventStreamUp(const DashboardActivityHeader& data) override {
    if (mEventStreamUpCallback != nullptr) {
      mEventStreamUpCallback(data);
    }
  }

  EventBitsUseageCallback mEventBitsUseageCallback;
  EventFollowerCallback mEventFollowerCallback;
  EventAutoHostStartCallback mEventAutoHostStartCallback;
  EventHostStartCallback mEventHostStartCallback;
  EventRaidingCallback mEventRaidingCallback;
  EventSubscriptionCallback mEventSubscriptionCallback;
  EventPrimeSubscriptionCallback mEventPrimeSubscriptionCallback;
  EventResubscriptionSharingCallback mEventResubscriptionSharingCallback;
  EventPrimeResubscriptionSharingCallback mEventPrimeResubscriptionSharingCallback;
  EventSubscriptionGiftingCommunityCallback mEventSubscriptionGiftingCommunityCallback;
  EventSubscriptionGiftingIndividualCallback mEventSubscriptionGiftingIndividualCallback;
  EventStreamUpCallback mEventStreamUpCallback;
};
}  // namespace

class DashboardActivityListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    InitializeModule(mCoreApi);
    AddModule(mCoreApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mListener = std::make_shared<TestDashboardActivityListener>();
  }

  template <class T>
  T HelperPubSubTest(std::function<void(T)>& listenerMethod, const std::string& payload) {
    bool receivedCallback = false;
    T result;

    listenerMethod = [&receivedCallback, &result](const T& data) {
      result = data;
      receivedCallback = true;
    };

    std::shared_ptr<IDashboardActivityStatus> status;
    TTV_ErrorCode ec = mCoreApi->CreateDashboardActivityStatus(mUserInfo.userId, 54321, mListener, status);
    EXPECT_TRUE(TTV_SUCCEEDED(ec));

    EXPECT_TRUE(WaitUntilResultWithPollTask(1000,
      [this]() { return mPubSubTestUtility.IsSubscribedToTopic("dashboard-activity-feed.54321"); },
      GetDefaultUpdateFunc()));

    mPubSubTestUtility.PushPubSubMessage("dashboard-activity-feed.54321", payload);

    EXPECT_TRUE(
      WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

    return result;
  }

  std::shared_ptr<CoreAPI> mCoreApi;

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestDashboardActivityListener> mListener;
};

TEST_F(DashboardActivityListenerTest, EventBitsUseage) {
  auto result = HelperPubSubTest<DashboardActivityBitsUsage>(mListener->mEventBitsUseageCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "bits_usage",)"
    R"("bits_amount": 5,)"
    R"("bits_anonymous": false,)"
    R"("bits_user": {)"
    R"("id": "2",)"
    R"("login": "3",)"
    R"("display_name": "4")"
    R"(}})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.user.userId, "2");
  EXPECT_EQ(result.user.login, "3");
  EXPECT_EQ(result.user.displayName, "4");
  EXPECT_EQ(result.amount, 5);
  EXPECT_EQ(result.anonymous, false);
}

TEST_F(DashboardActivityListenerTest, EventBitsUseageAnonymous) {
  auto result = HelperPubSubTest<DashboardActivityBitsUsage>(mListener->mEventBitsUseageCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "bits_usage",)"
    R"("bits_amount": 5,)"
    R"("bits_anonymous": true})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.user.userId, "");
  EXPECT_EQ(result.user.login, "");
  EXPECT_EQ(result.user.displayName, "");
  EXPECT_EQ(result.amount, 5);
  EXPECT_EQ(result.anonymous, true);
}

TEST_F(DashboardActivityListenerTest, EventFollower) {
  auto result = HelperPubSubTest<DashboardActivityFollow>(mListener->mEventFollowerCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "follow",)"
    R"("follower": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(}})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.follower.userId, "2");
  EXPECT_EQ(result.follower.login, "3");
  EXPECT_EQ(result.follower.displayName, "4");
}

TEST_F(DashboardActivityListenerTest, EventAutoHostStart) {
  auto result = HelperPubSubTest<DashboardActivityHost>(mListener->mEventAutoHostStartCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "auto_host_start",)"
    R"("host": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("hosting_viewer_count": 5)"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.host.userId, "2");
  EXPECT_EQ(result.host.login, "3");
  EXPECT_EQ(result.host.displayName, "4");
  EXPECT_EQ(result.viewerCount, 5);
}

TEST_F(DashboardActivityListenerTest, EventHostStart) {
  auto result = HelperPubSubTest<DashboardActivityHost>(mListener->mEventHostStartCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "host_start",)"
    R"("host": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("hosting_viewer_count": 5)"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.host.userId, "2");
  EXPECT_EQ(result.host.login, "3");
  EXPECT_EQ(result.host.displayName, "4");
  EXPECT_EQ(result.viewerCount, 5);
}

TEST_F(DashboardActivityListenerTest, EventRaiding) {
  auto result = HelperPubSubTest<DashboardActivityRaiding>(mListener->mEventRaidingCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "raiding",)"
    R"("raider": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("raiding_viewer_count": 5)"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.raider.userId, "2");
  EXPECT_EQ(result.raider.login, "3");
  EXPECT_EQ(result.raider.displayName, "4");
  EXPECT_EQ(result.viewerCount, 5);
}

TEST_F(DashboardActivityListenerTest, EventSubscription) {
  auto result = HelperPubSubTest<DashboardActivitySubscription>(mListener->mEventSubscriptionCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "subscription",)"
    R"("subscriber": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("subscription_tier": "5")"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
}

TEST_F(DashboardActivityListenerTest, EventPrimeSubscription) {
  auto result = HelperPubSubTest<DashboardActivitySubscription>(mListener->mEventPrimeSubscriptionCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "prime_subscription",)"
    R"("subscriber": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("subscription_tier": "5")"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
}

TEST_F(DashboardActivityListenerTest, EventResubscriptionSharingNoFragments) {
  auto result = HelperPubSubTest<DashboardActivityResubscriptionSharing>(mListener->mEventResubscriptionSharingCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "resubscription_sharing",)"
    R"("subscriber": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("subscription_tier": "5",)"
    R"("subscription_cumulative_tenure_months": 6,)"
    R"("subscription_custom_message_text": "7")"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
  EXPECT_EQ(result.cumulativeTenureMonths, 6);
  EXPECT_EQ(result.customMessage, "7");
  EXPECT_EQ(result.customMessageFragments.size(), 0);
}

TEST_F(DashboardActivityListenerTest, EventResubscriptionSharingWithFragments) {
  auto result = HelperPubSubTest<DashboardActivityResubscriptionSharing>(mListener->mEventResubscriptionSharingCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "resubscription_sharing",)"
    R"("subscriber": {)"
    R"(  "id": "2",)"
    R"(  "login" : "3",)"
    R"(  "display_name" : "4")"
    R"(},)"
    R"("subscription_tier": "5",)"
    R"("subscription_cumulative_tenure_months": 6,)"
    R"("subscription_custom_message_text": "7",)"
    R"("subscription_custom_message_fragments": [)"
    R"(  {)"
    R"(    "emoticon": {)"
    R"(      "id": "1",)"
    R"(      "set_id": "2")"
    R"(    },)"
    R"(    "text": "3")"
    R"(  },)"
    R"(  {)"
    R"(    "emoticon": {)"
    R"(      "id": "4",)"
    R"(      "set_id": "5")"
    R"(    },)"
    R"(    "text": "6")"
    R"(  })"
    R"(])"
    R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
  EXPECT_EQ(result.cumulativeTenureMonths, 6);
  EXPECT_EQ(result.customMessage, "7");
  ASSERT_EQ(result.customMessageFragments.size(), 2);
  EXPECT_EQ(result.customMessageFragments[0].emoticonId, "1");
  EXPECT_EQ(result.customMessageFragments[0].emoticonSetId, "2");
  EXPECT_EQ(result.customMessageFragments[0].text, "3");
  EXPECT_EQ(result.customMessageFragments[1].emoticonId, "4");
  EXPECT_EQ(result.customMessageFragments[1].emoticonSetId, "5");
  EXPECT_EQ(result.customMessageFragments[1].text, "6");
}

TEST_F(DashboardActivityListenerTest, EventPrimeResubscriptionSharingNoFragments) {
  auto result = HelperPubSubTest<DashboardActivityResubscriptionSharing>(
    mListener->mEventPrimeResubscriptionSharingCallback, R"({"id": "1",)"
                                                         R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                         R"("type": "prime_resubscription_sharing",)"
                                                         R"("subscriber": {)"
                                                         R"(  "id": "2",)"
                                                         R"(  "login" : "3",)"
                                                         R"(  "display_name" : "4")"
                                                         R"(},)"
                                                         R"("subscription_tier": "5",)"
                                                         R"("subscription_cumulative_tenure_months": 6,)"
                                                         R"("subscription_custom_message_text": "7")"
                                                         R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
  EXPECT_EQ(result.cumulativeTenureMonths, 6);
  EXPECT_EQ(result.customMessage, "7");
  EXPECT_EQ(result.customMessageFragments.size(), 0);
}

TEST_F(DashboardActivityListenerTest, EventPrimeResubscriptionSharingWithFragments) {
  auto result = HelperPubSubTest<DashboardActivityResubscriptionSharing>(
    mListener->mEventPrimeResubscriptionSharingCallback, R"({"id": "1",)"
                                                         R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                         R"("type": "prime_resubscription_sharing",)"
                                                         R"("subscriber": {)"
                                                         R"(  "id": "2",)"
                                                         R"(  "login" : "3",)"
                                                         R"(  "display_name" : "4")"
                                                         R"(},)"
                                                         R"("subscription_tier": "5",)"
                                                         R"("subscription_cumulative_tenure_months": 6,)"
                                                         R"("subscription_custom_message_text": "7",)"
                                                         R"("subscription_custom_message_fragments": [)"
                                                         R"(  {)"
                                                         R"(    "emoticon": {)"
                                                         R"(      "id": "1",)"
                                                         R"(      "set_id": "2")"
                                                         R"(    },)"
                                                         R"(    "text": "3")"
                                                         R"(  },)"
                                                         R"(  {)"
                                                         R"(    "emoticon": {)"
                                                         R"(      "id": "4",)"
                                                         R"(      "set_id": "5")"
                                                         R"(    },)"
                                                         R"(    "text": "6")"
                                                         R"(  })"
                                                         R"(])"
                                                         R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.subscriber.userId, "2");
  EXPECT_EQ(result.subscriber.login, "3");
  EXPECT_EQ(result.subscriber.displayName, "4");
  EXPECT_EQ(result.tier, "5");
  EXPECT_EQ(result.cumulativeTenureMonths, 6);
  EXPECT_EQ(result.customMessage, "7");
  ASSERT_EQ(result.customMessageFragments.size(), 2);
  EXPECT_EQ(result.customMessageFragments[0].emoticonId, "1");
  EXPECT_EQ(result.customMessageFragments[0].emoticonSetId, "2");
  EXPECT_EQ(result.customMessageFragments[0].text, "3");
  EXPECT_EQ(result.customMessageFragments[1].emoticonId, "4");
  EXPECT_EQ(result.customMessageFragments[1].emoticonSetId, "5");
  EXPECT_EQ(result.customMessageFragments[1].text, "6");
}

TEST_F(DashboardActivityListenerTest, EventSubscriptionGiftingCommunity) {
  auto result = HelperPubSubTest<DashboardActivitySubscriptionGiftingCommunity>(
    mListener->mEventSubscriptionGiftingCommunityCallback, R"({"id": "1",)"
                                                           R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                           R"("type": "subscription_gifting_community",)"
                                                           R"("subscription_gift_anonymous": false,)"
                                                           R"("subscription_gift_quantity": 2,)"
                                                           R"("subscription_gifter": {)"
                                                           R"(  "id": "3",)"
                                                           R"(  "login": "4",)"
                                                           R"(  "display_name": "5")"
                                                           R"(},)"
                                                           R"("subscription_gift_tier": "6")"
                                                           R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.quantity, 2);
  EXPECT_EQ(result.gifter.userId, "3");
  EXPECT_EQ(result.gifter.login, "4");
  EXPECT_EQ(result.gifter.displayName, "5");
  EXPECT_EQ(result.anonymous, false);
  EXPECT_EQ(result.tier, "6");
}

TEST_F(DashboardActivityListenerTest, EventSubscriptionGiftingCommunityAnonymous) {
  auto result = HelperPubSubTest<DashboardActivitySubscriptionGiftingCommunity>(
    mListener->mEventSubscriptionGiftingCommunityCallback, R"({"id": "1",)"
                                                           R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                           R"("type": "subscription_gifting_community",)"
                                                           R"("subscription_gift_anonymous": true,)"
                                                           R"("subscription_gift_quantity": 2,)"
                                                           R"("subscription_gift_tier": "3")"
                                                           R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.quantity, 2);
  EXPECT_EQ(result.anonymous, true);
  EXPECT_EQ(result.tier, "3");
}

TEST_F(DashboardActivityListenerTest, EventSubscriptionGiftingIndividual) {
  auto result = HelperPubSubTest<DashboardActivitySubscriptionGiftingIndividual>(
    mListener->mEventSubscriptionGiftingIndividualCallback, R"({"id": "1",)"
                                                            R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                            R"("type": "subscription_gifting_individual",)"
                                                            R"("subscription_gift_recipient": {)"
                                                            R"(  "id": "2",)"
                                                            R"(  "login": "3",)"
                                                            R"(  "display_name": "4")"
                                                            R"(},)"
                                                            R"("subscription_gift_tier": "5",)"
                                                            R"("subscription_gift_anonymous": false,)"
                                                            R"("subscription_gifter": {)"
                                                            R"(  "id": "6",)"
                                                            R"(  "login": "7",)"
                                                            R"(  "display_name": "8")"
                                                            R"(})"
                                                            R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.recipient.userId, "2");
  EXPECT_EQ(result.recipient.login, "3");
  EXPECT_EQ(result.recipient.displayName, "4");
  EXPECT_EQ(result.anonymous, false);
  EXPECT_EQ(result.tier, "5");
  EXPECT_EQ(result.gifter.userId, "6");
  EXPECT_EQ(result.gifter.login, "7");
  EXPECT_EQ(result.gifter.displayName, "8");
}

TEST_F(DashboardActivityListenerTest, EventSubscriptionGiftingIndividualAnonymous) {
  auto result = HelperPubSubTest<DashboardActivitySubscriptionGiftingIndividual>(
    mListener->mEventSubscriptionGiftingIndividualCallback, R"({"id": "1",)"
                                                            R"("timestamp": "2019-04-01T04:10:10Z",)"
                                                            R"("type": "subscription_gifting_individual",)"
                                                            R"("subscription_gift_recipient": {)"
                                                            R"(  "id": "2",)"
                                                            R"(  "login": "3",)"
                                                            R"(  "display_name": "4")"
                                                            R"(},)"
                                                            R"("subscription_gift_tier": "5",)"
                                                            R"("subscription_gift_anonymous": true)"
                                                            R"(})");

  EXPECT_EQ(result.header.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.header.timestamp), "2019-04-01T04:10:10Z");
  EXPECT_EQ(result.recipient.userId, "2");
  EXPECT_EQ(result.recipient.login, "3");
  EXPECT_EQ(result.recipient.displayName, "4");
  EXPECT_EQ(result.anonymous, true);
  EXPECT_EQ(result.tier, "5");
}

TEST_F(DashboardActivityListenerTest, EventStreamUp) {
  auto result = HelperPubSubTest<DashboardActivityHeader>(mListener->mEventStreamUpCallback,
    R"({"id": "1",)"
    R"("timestamp": "2019-04-01T04:10:10Z",)"
    R"("type": "stream_up")"
    R"(})");

  EXPECT_EQ(result.id, "1");
  EXPECT_EQ(UnixTimestampToRFC3339String(result.timestamp), "2019-04-01T04:10:10Z");
}
