/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/eventqueue.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class EventQueueTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();
    mEventQueue = std::make_shared<EventQueue>("TestEventQueue");
  }

  std::shared_ptr<EventQueue> mEventQueue;
  int32_t mTasksRun = 0;
};
}  // namespace

TEST_F(EventQueueTest, SimpleInserts) {
  mEventQueue->InsertTask({[this] { mTasksRun++; }});
  mEventQueue->InsertTask({[this] { mTasksRun++; }, 0});
  mEventQueue->InsertTask({[this] { mTasksRun++; }, "Test 1"});
  mEventQueue->InsertTask({[this] { mTasksRun++; }, 0, "Test 3"});

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(0)) {
    count++;
  }
  ASSERT_TRUE(count == 4);
  ASSERT_TRUE(mTasksRun == 4);
}

TEST_F(EventQueueTest, SimpleInsertsNoTimeout) {
  // WaitForEvent()
  for (int i = 0; i < 100; i++) {
    mEventQueue->InsertTask({[this] { mTasksRun++; }});
  }
  for (int i = 0; i < 100; i++) {
    mEventQueue->WaitForEvent();
  }

  ASSERT_TRUE(mTasksRun == 100);
  mTasksRun = 0;

  // WaitForEventWithTimeout(std::numeric_limits<uint64_t>::max())
  for (int i = 0; i < 100; i++) {
    mEventQueue->InsertTask({[this] { mTasksRun++; }});
  }
  for (int i = 0; i < 100; i++) {
    mEventQueue->WaitForEventWithTimeout(std::numeric_limits<uint64_t>::max());
  }

  ASSERT_TRUE(mTasksRun == 100);
  mTasksRun = 0;

  // WaitForEventWithTimeout(0)
  for (int i = 0; i < 100; i++) {
    mEventQueue->InsertTask({[this] { mTasksRun++; }});
  }
  for (int i = 0; i < 100; i++) {
    mEventQueue->WaitForEventWithTimeout(0);  // these should all trigger immediately
  }

  ASSERT_TRUE(mTasksRun == 100);
  mTasksRun = 0;

  // WaitForEventWithTimeout(100)
  for (int i = 0; i < 100; i++) {
    mEventQueue->InsertTask({[this] { mTasksRun++; }});
  }
  for (int i = 0; i < 100; i++) {
    mEventQueue->WaitForEventWithTimeout(100);  // these should all trigger immediately
  }

  ASSERT_TRUE(mTasksRun == 100);
  mTasksRun = 0;
}

TEST_F(EventQueueTest, TaskParams) {
  // Test the sorted order of the output (with some cancellation)
  std::vector<TaskId> tasksRun;

  TaskId result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(1); }, 40});
  ASSERT_TRUE(result == 1);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(2); }});
  ASSERT_TRUE(result == 2);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(3); }, "Task #3"});
  ASSERT_TRUE(result == 3);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(4); }, 40, "Task #5"});
  ASSERT_TRUE(result == 4);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(5); }, 10});
  ASSERT_TRUE(result == 5);

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(50)) {
    count++;
  }

  ASSERT_TRUE(count == 5);
  ASSERT_TRUE(tasksRun.size() == 5);

  std::array<int, 5> arr{{2, 3, 5, 1, 4}};

  for (int i = 0; i < 5; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }
}

TEST_F(EventQueueTest, Stress) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        mEventQueue->InsertTask({[this] { mTasksRun++; }});
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), 20);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(10)) {
    count++;
  }

  ASSERT_TRUE(count == numThreads * tasksPerThread);
  ASSERT_TRUE(mTasksRun == numThreads * tasksPerThread);
}

TEST_F(EventQueueTest, StressCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        TaskId id = mEventQueue->InsertTask({[this] { mTasksRun++; }});
        ASSERT_TRUE(mEventQueue->RemoveTask(id));
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(10)) {
    count++;
  }

  ASSERT_TRUE(count == 0);
  ASSERT_TRUE(mTasksRun == 0);
}

TEST_F(EventQueueTest, StressHalfCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        TaskId id = mEventQueue->InsertTask({[this] { mTasksRun++; }});
        if (id % 2 == 0) {
          ASSERT_TRUE(mEventQueue->RemoveTask(id));
        }
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(10)) {
    count++;
  }

  int total = (numThreads * tasksPerThread + 1) / 2;

  ASSERT_TRUE(count == total);
  ASSERT_TRUE(mTasksRun == total);
}

TEST_F(EventQueueTest, SortedOrder) {
  // Test the sorted order of the output (with some cancellation)
  std::vector<TaskId> tasksRun;

  TaskId result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(1); }, 40});
  ASSERT_TRUE(result == 1);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(2); }, 40});
  ASSERT_TRUE(result == 2);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(3); }, 10});
  ASSERT_TRUE(result == 3);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(4); }, 0});
  ASSERT_TRUE(result == 4);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(5); }, 10});
  ASSERT_TRUE(result == 5);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(6); }, 20});
  ASSERT_TRUE(result == 6);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(7); }, 0});
  ASSERT_TRUE(result == 7);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(8); }, 0});
  ASSERT_TRUE(result == 8);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(9); }, 40});
  ASSERT_TRUE(result == 9);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(10); }, 30});
  ASSERT_TRUE(result == 10);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(11); }, 30});
  ASSERT_TRUE(result == 11);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(12); }, 30});
  ASSERT_TRUE(result == 12);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(13); }, 10});
  ASSERT_TRUE(result == 13);

  ASSERT_TRUE(mEventQueue->RemoveTask(1));

  ASSERT_TRUE(mEventQueue->RemoveTask(8));

  ASSERT_TRUE(mEventQueue->RemoveTask(11));

  int count = 0;
  while (mEventQueue->WaitForEventWithTimeout(50)) {
    count++;
  }

  ASSERT_TRUE(count == 10);
  ASSERT_TRUE(tasksRun.size() == 10);

  std::array<int, 10> arr{{4, 7, 3, 5, 13, 6, 10, 12, 2, 9}};

  for (int i = 0; i < 10; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }
}

TEST_F(EventQueueTest, WaitTimeout) {
  // Test waits properly working with an insert with delay

  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(10));

  TaskId result = mEventQueue->InsertTask({[this] { mTasksRun++; }, 50});
  ASSERT_TRUE(result == 1);

  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(20));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(50));
  ASSERT_EQ(mTasksRun, 1);
}

TEST_F(EventQueueTest, Cancel) {
  // Test various carious operations

  ASSERT_FALSE(mEventQueue->RemoveTask(0));
  ASSERT_FALSE(mEventQueue->RemoveTask(1));
  ASSERT_FALSE(mEventQueue->RemoveTask(2));
  ASSERT_FALSE(mEventQueue->RemoveTask(100));

  TaskId result = mEventQueue->InsertTask({[this] { mTasksRun++; }, 0});
  ASSERT_TRUE(result == 1);

  ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(0));
  ASSERT_FALSE(mEventQueue->RemoveTask(1));

  result = mEventQueue->InsertTask({[this] { mTasksRun++; }, 0});
  ASSERT_TRUE(result == 2);

  ASSERT_TRUE(mEventQueue->RemoveTask(2));
  ASSERT_FALSE(mEventQueue->RemoveTask(2));
  ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));

  ASSERT_EQ(mTasksRun, 1);
}

TEST_F(EventQueueTest, MultiThreadWaitNoTimeout) {
  // Test multi-threaded scenario only using WaitForEvent() with no timeouts
  std::shared_ptr<IThread> bgThread;
  std::vector<TaskId> tasksRun;

  auto bgThreadProc = [this, &tasksRun]() {
    ttv::Sleep(20);

    TaskId result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(1); }, 20});
    ASSERT_TRUE(result == 1);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(2); }, 30});
    ASSERT_TRUE(result == 2);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(3); }, 40});
    ASSERT_TRUE(result == 3);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(4); }, 20});
    ASSERT_TRUE(result == 4);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(5); }, 30});
    ASSERT_TRUE(result == 5);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(6); }, 20});
    ASSERT_TRUE(result == 6);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(7); }, 0});
    ASSERT_TRUE(result == 7);

    result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(8); }, 10});
    ASSERT_TRUE(result == 8);

    ttv::Sleep(10);

    ASSERT_FALSE(mEventQueue->RemoveTask(7));
    ASSERT_TRUE(mEventQueue->RemoveTask(2));
  };

  std::shared_ptr<IThread> thread;
  CreateThread(bgThreadProc, "bg thread", thread);
  thread->Run();

  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();
  mEventQueue->WaitForEvent();

  thread->Join();
  thread.reset();

  ASSERT_TRUE(tasksRun.size() == 7);

  std::array<int, 7> arr{{7, 8, 1, 4, 6, 5, 3}};

  for (int i = 0; i < 7; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }
}

TEST_F(EventQueueTest, MultiThreadedWaitWithTimeout) {
  // Test multi-threaded scenario using WaitForEventWithTimeout()
  std::shared_ptr<IThread> bgThread;
  std::vector<TaskId> tasksRun;

  auto bgThreadProc = [this]() {
    ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(10));
    ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
    ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(100));
    ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(0));
    ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(50));
    ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(100));
    ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(100));
    ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(100));
    ASSERT_TRUE(mEventQueue->WaitForEventWithTimeout(100));
    ASSERT_FALSE(mEventQueue->WaitForEventWithTimeout(100));
  };

  std::shared_ptr<IThread> thread;
  CreateThread(bgThreadProc, "bg thread", thread);
  thread->Run();

  ttv::Sleep(15);

  TaskId result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(1); }, 30});
  ASSERT_TRUE(result == 1);

  ttv::Sleep(50);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(2); }, 30});
  ASSERT_TRUE(result == 2);

  ASSERT_FALSE(mEventQueue->RemoveTask(1));
  ASSERT_TRUE(mEventQueue->RemoveTask(2));

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(3); }, 110});
  ASSERT_TRUE(result == 3);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(4); }, 100});
  ASSERT_TRUE(result == 4);

  ttv::Sleep(50);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(5); }, 0});
  ASSERT_TRUE(result == 5);

  result = mEventQueue->InsertTask({[&tasksRun] { tasksRun.emplace_back(6); }, 0});
  ASSERT_TRUE(result == 6);

  thread->Join();
  thread.reset();

  ASSERT_TRUE(tasksRun.size() == 5);

  std::array<int, 5> arr{{1, 5, 6, 4, 3}};

  for (int i = 0; i < 5; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }
}
