/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/trackingcontext.h"

#include "gtest/gtest.h"

using namespace ttv::test;

namespace {
bool CheckTrackingValue(const ttv::TrackingValue& value, nullptr_t /*expected*/) {
  return value.GetType() == ttv::TrackingValue::Type::Null;
}

bool CheckTrackingValue(const ttv::TrackingValue& value, bool expected) {
  return (value.GetType() == ttv::TrackingValue::Type::Boolean) && (value.GetBooleanValue() == expected);
}

bool CheckTrackingValue(const ttv::TrackingValue& value, int expected) {
  return (value.GetType() == ttv::TrackingValue::Type::Integer) && (value.GetIntegerValue() == expected);
}

bool CheckTrackingValue(const ttv::TrackingValue& value, double expected) {
  return (value.GetType() == ttv::TrackingValue::Type::Double) && (abs(value.GetDoubleValue() - expected) < 0.001);
}

bool CheckTrackingValue(const ttv::TrackingValue& value, const char* expected) {
  return (value.GetType() == ttv::TrackingValue::Type::String) && (value.GetStringValue() == expected);
}

class TestEventTracker : public ttv::IEventTracker {
 public:
  struct Event {
    std::string name;
    std::map<std::string, ttv::TrackingValue> properties;

    template <typename Value>
    bool CheckForProperty(const std::string& key, Value&& value) const {
      auto iter = properties.find(key);
      if (iter == properties.end()) {
        return false;
      }

      return CheckTrackingValue(iter->second, std::forward<Value>(value));
    }
  };

  const std::vector<Event>& GetEvents() { return mEvents; }

  virtual TTV_ErrorCode TrackEvent(
    const std::string& eventName, const std::map<std::string, ttv::TrackingValue>& properties) override {
    Event newEvent;
    newEvent.name = eventName;
    newEvent.properties = properties;

    mEvents.push_back(std::move(newEvent));

    return TTV_EC_SUCCESS;
  }

 private:
  std::vector<Event> mEvents;
};

class EventTrackerTest : public SdkBaseTest {
  virtual void SetUpStubs() override {
    mPreviousEventTracker = ttv::GetEventTracker();
    mTestEventTracker = std::make_shared<TestEventTracker>();
    ttv::SetEventTracker(mTestEventTracker);
  }

  virtual void TearDownStubs() override { ttv::SetEventTracker(mPreviousEventTracker); }

 protected:
  std::shared_ptr<TestEventTracker> mTestEventTracker;

 private:
  std::shared_ptr<ttv::IEventTracker> mPreviousEventTracker;
};
}  // namespace

TEST_F(EventTrackerTest, TestEventTracking) {
  // Test rvalues
  ASSERT_TRUE(TTV_SUCCEEDED(ttv::TrackEvent("FirstEvent", {{"boolProperty", true}, {"integerProperty", 777}})));

  // Test lvalues
  std::string eventName = "SecondEvent";
  std::map<std::string, ttv::TrackingValue> properties;
  properties.emplace("doubleProperty", 3.0);
  properties.emplace("stringProperty", "stringValue");
  properties.emplace("nullProperty", ttv::TrackingValue{});

  ASSERT_TRUE(TTV_SUCCEEDED(ttv::TrackEvent(eventName, properties)));

  const auto& receivedEvents = mTestEventTracker->GetEvents();

  ASSERT_EQ(receivedEvents.size(), 2);

  const auto& firstEvent = receivedEvents[0];
  ASSERT_EQ(firstEvent.name, "FirstEvent");
  ASSERT_EQ(firstEvent.properties.size(), 2);

  ASSERT_TRUE(firstEvent.CheckForProperty("boolProperty", true));
  ASSERT_TRUE(firstEvent.CheckForProperty("integerProperty", 777));

  const auto& secondEvent = receivedEvents[1];
  ASSERT_EQ(secondEvent.name, "SecondEvent");
  ASSERT_EQ(secondEvent.properties.size(), 3);

  ASSERT_TRUE(secondEvent.CheckForProperty("doubleProperty", 3.0));
  ASSERT_TRUE(secondEvent.CheckForProperty("stringProperty", "stringValue"));
  ASSERT_TRUE(secondEvent.CheckForProperty("nullProperty", nullptr));
}

TEST_F(EventTrackerTest, TestRootTrackingContext) {
  auto trackingContext = std::make_shared<ttv::TrackingContext>();
  trackingContext->SetProperty("shared_integer_property", 777);

  trackingContext->SetProperties({{"shared_boolean_property", true}, {"shared_double_property", 3.0}});

  ASSERT_TRUE(
    TTV_SUCCEEDED(trackingContext->TrackEvent("first_event", {{"property_one", "uno"}, {"property_two", "dos"}})));

  ASSERT_TRUE(
    TTV_SUCCEEDED(trackingContext->TrackEvent("second_event", {{"property_uno", "one"}, {"property_dos", "two"}})));

  const auto& receivedEvents = mTestEventTracker->GetEvents();
  ASSERT_EQ(receivedEvents.size(), 2);

  const auto& firstEvent = receivedEvents[0];
  ASSERT_EQ(firstEvent.name, "first_event");
  ASSERT_EQ(firstEvent.properties.size(), 5);

  ASSERT_TRUE(firstEvent.CheckForProperty("shared_integer_property", 777));
  ASSERT_TRUE(firstEvent.CheckForProperty("shared_boolean_property", true));
  ASSERT_TRUE(firstEvent.CheckForProperty("shared_double_property", 3.0));
  ASSERT_TRUE(firstEvent.CheckForProperty("property_one", "uno"));
  ASSERT_TRUE(firstEvent.CheckForProperty("property_two", "dos"));

  const auto& secondEvent = receivedEvents[1];
  ASSERT_EQ(secondEvent.name, "second_event");
  ASSERT_EQ(secondEvent.properties.size(), 5);

  ASSERT_TRUE(secondEvent.CheckForProperty("shared_integer_property", 777));
  ASSERT_TRUE(secondEvent.CheckForProperty("shared_boolean_property", true));
  ASSERT_TRUE(secondEvent.CheckForProperty("shared_double_property", 3.0));
  ASSERT_TRUE(secondEvent.CheckForProperty("property_uno", "one"));
  ASSERT_TRUE(secondEvent.CheckForProperty("property_dos", "two"));
}

TEST_F(EventTrackerTest, TestNestedTrackingContexts) {
  auto rootTrackingContext = std::make_shared<ttv::TrackingContext>();
  rootTrackingContext->SetProperty("root_shared_property", "root_value");

  auto firstChildContext = std::make_shared<ttv::TrackingContext>(rootTrackingContext);
  firstChildContext->SetProperty("first_shared_property", 12345);

  auto secondChildContext = std::make_shared<ttv::TrackingContext>(rootTrackingContext);
  secondChildContext->SetProperty("second_shared_property", 12.345);

  ASSERT_TRUE(TTV_SUCCEEDED(rootTrackingContext->TrackEvent("root_event", {{"root_distinct_property", true}})));

  ASSERT_TRUE(
    TTV_SUCCEEDED(firstChildContext->TrackEvent("first_child_event", {{"first_child_distinct_property", nullptr}})));

  ASSERT_TRUE(TTV_SUCCEEDED(
    secondChildContext->TrackEvent("second_child_event", {{"second_child_distinct_property", "prop_value"}})));

  const auto& receivedEvents = mTestEventTracker->GetEvents();
  ASSERT_EQ(receivedEvents.size(), 3);

  const auto& firstEvent = receivedEvents[0];
  ASSERT_EQ(firstEvent.name, "root_event");
  ASSERT_EQ(firstEvent.properties.size(), 2);

  ASSERT_TRUE(firstEvent.CheckForProperty("root_shared_property", "root_value"));
  ASSERT_TRUE(firstEvent.CheckForProperty("root_distinct_property", true));

  const auto& secondEvent = receivedEvents[1];
  ASSERT_EQ(secondEvent.name, "first_child_event");
  ASSERT_EQ(secondEvent.properties.size(), 3);

  ASSERT_TRUE(secondEvent.CheckForProperty("root_shared_property", "root_value"));
  ASSERT_TRUE(secondEvent.CheckForProperty("first_shared_property", 12345));
  ASSERT_TRUE(secondEvent.CheckForProperty("first_child_distinct_property", nullptr));

  const auto& thirdEvent = receivedEvents[2];
  ASSERT_EQ(thirdEvent.name, "second_child_event");
  ASSERT_EQ(thirdEvent.properties.size(), 3);

  ASSERT_TRUE(thirdEvent.CheckForProperty("root_shared_property", "root_value"));
  ASSERT_TRUE(thirdEvent.CheckForProperty("second_shared_property", 12.345));
  ASSERT_TRUE(thirdEvent.CheckForProperty("second_child_distinct_property", "prop_value"));
}

TEST_F(EventTrackerTest, TestTrackingContextPropertyOverride) {
  auto rootTrackingContext = std::make_shared<ttv::TrackingContext>();
  rootTrackingContext->SetProperty("shared_property", 12345);

  auto childContext = std::make_shared<ttv::TrackingContext>(rootTrackingContext);
  childContext->SetProperty("shared_property", 54321);

  ASSERT_TRUE(TTV_SUCCEEDED(rootTrackingContext->TrackEvent("first_event", {})));
  ASSERT_TRUE(TTV_SUCCEEDED(childContext->TrackEvent("second_event", {})));
  ASSERT_TRUE(TTV_SUCCEEDED(childContext->TrackEvent("third_event", {{"shared_property", "override"}})));

  const auto& receivedEvents = mTestEventTracker->GetEvents();
  ASSERT_EQ(receivedEvents.size(), 3);

  const auto& firstEvent = receivedEvents[0];
  ASSERT_EQ(firstEvent.name, "first_event");
  ASSERT_EQ(firstEvent.properties.size(), 1);

  ASSERT_TRUE(firstEvent.CheckForProperty("shared_property", 12345));

  const auto& secondEvent = receivedEvents[1];
  ASSERT_EQ(secondEvent.name, "second_event");
  ASSERT_EQ(secondEvent.properties.size(), 1);

  // Child context's shared property should override the parent's shared property with the same key.
  ASSERT_TRUE(secondEvent.CheckForProperty("shared_property", 54321));

  const auto& thirdEvent = receivedEvents[2];
  ASSERT_EQ(thirdEvent.name, "third_event");
  ASSERT_EQ(thirdEvent.properties.size(), 1);

  // The explicitly set property should override any shared properties with the same keyu.
  ASSERT_TRUE(thirdEvent.CheckForProperty("shared_property", "override"));
}
