/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/
#include "fixtures/repositorytest.h"

#include "testutilities.h"
#include "twitchsdk/core/component.h"

using namespace ttv;

RepositoryTest::RepositoryTest() {
  mHttpRequest = std::make_shared<TestHttpRequest>();
  ttv::SetHttpRequest(std::static_pointer_cast<ttv::HttpRequest>(mHttpRequest));

  mTaskRunner = std::make_shared<ttv::TaskRunner>();
  mComponentContainer = std::make_shared<ComponentContainer>();
  mComponentContainer->SetTaskRunner(mTaskRunner);
  ttv::test::InitializeComponent(mComponentContainer);
  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> checkForInit = [this]() {
    return mComponentContainer->GetState() == IComponent::State::Initialized;
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(500, checkForInit, updateFunction));
}

void RepositoryTest::AddComponent(std::shared_ptr<ttv::Component> component) {
  mComponentContainer->AddComponent(component);
  component->SetTaskRunner(mTaskRunner);
  ttv::test::InitializeComponent(component);

  std::function<void()> updateFunction = [this] { Update(); };

  std::function<bool()> checkForInit = [component]() {
    return component->GetState() == IComponent::State::Initialized;
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(500, checkForInit, updateFunction));
}

RepositoryTest::~RepositoryTest() {
  ttv::test::ShutdownTaskRunner(mTaskRunner);
  ttv::test::ShutdownComponent(mComponentContainer);

  //  When refactoring to handle generic repository testing, include post-shutdown methods
  //  EXPECT_TRUE(TTV_FAILED(mChannelRepository->LookupChannelInfo("twitch", true, nullptr)));
}

void RepositoryTest::Update() {
  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  if (mComponentContainer != nullptr) {
    mComponentContainer->Update();
  }
}
