/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/igenericsubscriberlistener.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class TestGenericSubscriberListener : public IGenericSubscriberListener {
 public:
  ~TestGenericSubscriberListener() override = default;

  void EventTopicData(const std::string& data) override { mEventTopicData(data); }

  std::function<void(const std::string)> mEventTopicData;
};
}  // namespace

class GenericSubscriberListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    InitializeModule(mCoreApi);
    AddModule(mCoreApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mListener = std::make_shared<TestGenericSubscriberListener>();
  }

  template <class T>
  T HelperPubSubTest(
    std::function<void(T)>& listenerMethod, const std::string& subscriberTopic, const std::string& payload) {
    bool receivedCallback = false;
    T result;

    listenerMethod = [&receivedCallback, &result](const T& data) {
      result = data;
      receivedCallback = true;
    };

    std::shared_ptr<IGenericSubscriberStatus> status;
    TTV_ErrorCode ec = mCoreApi->CreateGenericSubscriberStatus(mUserInfo.userId, subscriberTopic, mListener, status);
    EXPECT_TRUE(TTV_SUCCEEDED(ec));

    EXPECT_TRUE(WaitUntilResultWithPollTask(1000,
      [this, subscriberTopic]() { return mPubSubTestUtility.IsSubscribedToTopic(subscriberTopic); },
      GetDefaultUpdateFunc()));

    mPubSubTestUtility.PushPubSubMessage(subscriberTopic.c_str(), payload);

    EXPECT_TRUE(
      WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

    return result;
  }

  std::shared_ptr<CoreAPI> mCoreApi;

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestGenericSubscriberListener> mListener;
};

TEST_F(GenericSubscriberListenerTest, EventTopicDataValidJson) {
  std::string expect = R"({"string": "test_string",)"
                       R"("timestamp": "2019-04-01T04:10:10Z",)"
                       R"("double": 0.004,)"
                       R"("array": [)"
                       R"(  {)"
                       R"(    "obj1": {)"
                       R"(      "obj1_1": "1",)"
                       R"(      "obj1_2": "2")"
                       R"(    })"
                       R"(  },{)"
                       R"(    "obj2": {)"
                       R"(      "obj2_1": "1",)"
                       R"(      "obj2_2": "2")"
                       R"(    })"
                       R"(  })"
                       R"(]})";

  auto result = HelperPubSubTest<std::string>(mListener->mEventTopicData, std::string("random_topic.12345"), expect);

  json::Value input;
  json::Value output;

  json::Reader().parse(expect, input);
  json::Reader().parse(result, output);

  EXPECT_EQ(input, output);
}

TEST_F(GenericSubscriberListenerTest, EventTopicDataMalformedJson) {
  std::string expect = R"({"string": "test_string",)"
                       R"("timestamp": "2019-04-01T04:10:10Z",)"
                       R"("double": 0.004,)"
                       R"("array": [)"
                       R"( : {)"
                       R"(  '  "obj1": {)"
                       R"(   "   "obj1_1": "1",)"
                       R"(   ~   "obj1_2": "2")"
                       R"(  1  })"
                       R"(  },)"
                       R"(    "obj2": {)"
                       R"(      "obj2_1": "1",)"
                       R"(   3412   "obj2_2": "2")"
                       R"(    })"
                       R"(  })"
                       R"(]}Malformed)";

  auto result = HelperPubSubTest<std::string>(mListener->mEventTopicData, std::string("random_topic.12345"), expect);

  json::Value input;
  json::Value output;

  json::Reader().parse(expect, input);
  json::Reader().parse(result, output);

  EXPECT_EQ(input, output);
}

TEST_F(GenericSubscriberListenerTest, EventTopicDataEmpty) {
  std::string expect = "";

  auto result = HelperPubSubTest<std::string>(mListener->mEventTopicData, std::string("random_topic.12345"), expect);

  json::Value input;
  json::Value output;

  json::Reader().parse(expect, input);
  json::Reader().parse(result, output);

  EXPECT_EQ(input, output);
}
