/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"

using namespace ttv;
using namespace ttv::test;

TEST(JsonSerialization, ParseUserInfo) {
  UserInfo userInfo;
  bool success = ttv::json::ToObject(
    "{"
    "\"_id\":9001,"
    "\"name\":\"testuser\","
    "\"display_name\":\"TestUser\","
    "\"bio\":\"This is a bio.\","
    "\"created_at\":\"2016-09-14T23:47:47Z\","
    "\"logo\":\"https://logo.com\""
    "}",
    userInfo);

  EXPECT_TRUE(success);

  EXPECT_EQ(userInfo.userId, 9001);
  EXPECT_EQ(userInfo.userName, "testuser");
  EXPECT_EQ(userInfo.displayName, "TestUser");
  EXPECT_EQ(userInfo.bio, "This is a bio.");
  EXPECT_EQ(userInfo.createdTimestamp, 1473896867);
  EXPECT_EQ(userInfo.logoImageUrl, "https://logo.com");
}

TEST(JsonSerialization, ParseUserInfo_IncorrectTypes) {
  UserInfo userInfo;
  bool success = ttv::json::ToObject(
    "{"
    "\"_id\":-9001,"
    "\"name\":\"testuser\","
    "\"display_name\":\"TestUser\","
    "\"bio\":\"This is a bio.\","
    "\"created_at\":\"2016-09-14T23:47:47Z\","
    "\"logo\":\"https://logo.com\""
    "}",
    userInfo);

  EXPECT_FALSE(success);
}

TEST(JsonSerialization, ParseUserInfoMissingOptionalFields) {
  UserInfo userInfo;
  bool success = ttv::json::ToObject(
    "{"
    "\"_id\":9001,"
    "\"name\":\"testuser\","
    "\"display_name\":\"TestUser\","
    "\"created_at\":\"2016-09-14T23:47:47Z\","
    "\"logo\":\"https://logo.com\""
    "}",
    userInfo);

  EXPECT_TRUE(success);

  EXPECT_EQ(userInfo.userId, 9001);
  EXPECT_EQ(userInfo.userName, "testuser");
  EXPECT_EQ(userInfo.displayName, "TestUser");
  EXPECT_EQ(userInfo.createdTimestamp, 1473896867);
  EXPECT_EQ(userInfo.logoImageUrl, "https://logo.com");
}

TEST(JsonSerialization, ParseUserInfoMissingRequiredFields) {
  UserInfo userInfo;
  bool success = ttv::json::ToObject(
    "{"
    "\"name\":\"testuser\","
    "\"display_name\":\"TestUser\","
    "\"bio\":\"This is a bio.\","
    "\"created_at\":\"2016-09-14T23:47:47Z\","
    "\"logo\":\"https://logo.com\""
    "}",
    userInfo);

  EXPECT_FALSE(success);
}

TEST(JsonSerialization, EmitUserInfo) {
  UserInfo userInfo;
  userInfo.userId = 1234;
  userInfo.userName = "testuser";
  userInfo.displayName = "TestUser";
  userInfo.bio = "This is a bio.";
  userInfo.logoImageUrl = "https://logo.com";
  userInfo.createdTimestamp = 1473896867;

  ttv::json::Value value;
  ttv::json::ToJson(userInfo, value);

  EXPECT_EQ(value["_id"].asUInt(), 1234);
  EXPECT_EQ(value["name"], "testuser");
  EXPECT_EQ(value["display_name"], "TestUser");
  EXPECT_EQ(value["bio"], "This is a bio.");
  EXPECT_EQ(value["logo"], "https://logo.com");

  Timestamp timestamp;
  EXPECT_TRUE(ttv::RFC3339TimeToUnixTimestamp(value["created_at"].asString(), timestamp));
  EXPECT_EQ(timestamp, 1473896867);
}

namespace {
struct ChildObject {
  uint32_t myNumber;
  std::string myString;
  bool myBool;
  uint32_t myNestedValue;
};

struct ParentObject {
  uint32_t myNumber;
  std::string myString;
  double myFloat;
  ChildObject child;
};

struct CollectionObject {
  uint32_t myNumber;
  std::string myString;
  std::vector<ChildObject> children;
};

struct ChildObjectDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field("myNumber", object.myNumber),
      ttv::json::make_field("myString", object.myString), ttv::json::make_field("myBool", object.myBool),
      ttv::json::make_field<ttv::json::OptionalField>(
        ttv::json::MakeKeyPath("key1", "key2", "key3"), object.myNestedValue));
  }
};

struct ParentObjectDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field("myNumber", object.myNumber),
      ttv::json::make_field("myString", object.myString), ttv::json::make_field("myFloat", object.myFloat),
      ttv::json::make_field("child", object.child));
  }
};

struct CollectionObjectDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field("myNumber", object.myNumber),
      ttv::json::make_field("myString", object.myString), ttv::json::make_field("children", object.children));
  }
};
}  // namespace

template <>
struct ttv::json::DefaultSchemaProvider<ChildObject> {
  using Type = ObjectSchema<ChildObjectDescription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ParentObject> {
  using Type = ObjectSchema<ParentObjectDescription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<CollectionObject> {
  using Type = ObjectSchema<CollectionObjectDescription>;
};

TEST(JsonSerialization, NestedObjects) {
  json::Value root;
  root["myNumber"] = 12345;
  root["myString"] = "parentString";
  root["myFloat"] = 3.6;

  auto& child = root["child"];
  child["myNumber"] = 6789;
  child["myString"] = "childString";
  child["myBool"] = true;

  ParentObject object;
  bool parsed = ttv::json::ToObject(root, object);

  EXPECT_TRUE(parsed);

  EXPECT_EQ(object.myNumber, 12345);
  EXPECT_EQ(object.myString, "parentString");
  EXPECT_DOUBLE_EQ(object.myFloat, 3.6);
  EXPECT_EQ(object.child.myNumber, 6789);
  EXPECT_EQ(object.child.myString, "childString");
  EXPECT_TRUE(object.child.myBool);
}

TEST(JsonSerialization, KeyPath) {
  json::Value root;

  ChildObject object;
  object.myNumber = 9876;
  object.myString = "childString";
  object.myBool = true;
  object.myNestedValue = 123;

  bool emitted = ttv::json::ToJson(object, root);

  ASSERT_TRUE(emitted);

  ASSERT_EQ(root["myNumber"].asUInt(), 9876);
  ASSERT_EQ(root["myString"].asString(), "childString");
  ASSERT_TRUE(root["myBool"].asBool());
  ASSERT_EQ(root["key1"]["key2"]["key3"].asUInt(), 123);

  ChildObject object2;
  bool parsed = ttv::json::ToObject(root, object2);

  ASSERT_TRUE(parsed);

  ASSERT_EQ(object2.myNumber, 9876);
  ASSERT_EQ(object2.myString, "childString");
  ASSERT_TRUE(object2.myBool);
  ASSERT_EQ(object2.myNestedValue, 123);
}

TEST(JsonSerialization, CollectionObjects) {
  json::Value root;
  root["myNumber"] = 12345;
  root["myString"] = "parentString";

  json::Value& children = root["children"];

  json::Value firstChild;
  firstChild["myNumber"] = 6789;
  firstChild["myString"] = "firstChild";
  firstChild["myBool"] = true;

  children.append(firstChild);

  json::Value secondChild;
  secondChild["myNumber"] = 54321;
  secondChild["myString"] = "secondChild";
  secondChild["myBool"] = false;

  children.append(secondChild);

  json::Value thirdChild;
  thirdChild["myNumber"] = 9876;
  thirdChild["myString"] = "thirdChild";
  thirdChild["myBool"] = true;

  children.append(thirdChild);

  CollectionObject object;
  bool parsed = ttv::json::ToObject(root, object);

  EXPECT_TRUE(parsed);

  EXPECT_EQ(object.myNumber, 12345);
  EXPECT_EQ(object.myString, "parentString");

  EXPECT_EQ(object.children.size(), 3);
  EXPECT_EQ(object.children[0].myNumber, 6789);
  EXPECT_EQ(object.children[0].myString, "firstChild");
  EXPECT_TRUE(object.children[0].myBool);
  EXPECT_EQ(object.children[1].myNumber, 54321);
  EXPECT_EQ(object.children[1].myString, "secondChild");
  EXPECT_FALSE(object.children[1].myBool);
  EXPECT_EQ(object.children[2].myNumber, 9876);
  EXPECT_EQ(object.children[2].myString, "thirdChild");
  EXPECT_TRUE(object.children[2].myBool);
}

namespace {
enum class TestEnum { First, Second, Third };

struct TestEnumDescription {
  static auto EnumMap() {
    return std::make_tuple(ttv::json::make_enum_mapping("first", TestEnum::First),
      ttv::json::make_enum_mapping("second", TestEnum::Second), ttv::json::make_enum_mapping("third", TestEnum::Third));
  }
};

struct TestEnumParent {
  TestEnum uno;
  TestEnum dos;
  TestEnum tres;
};

struct TestEnumParentDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field("uno", object.uno), ttv::json::make_field("dos", object.dos),
      ttv::json::make_field<ttv::json::RequiredField>("tres", object.tres));
  }
};

enum class TestFallbackEnum { First, Second, Third, Unknown };

struct TestFallbackEnumDescription {
  static auto EnumMap() {
    return std::make_tuple(ttv::json::make_enum_mapping("first", TestFallbackEnum::First),
      ttv::json::make_enum_mapping("second", TestFallbackEnum::Second),
      ttv::json::make_enum_mapping("third", TestFallbackEnum::Third));
  }

  static auto GetFallbackValue() { return TestFallbackEnum::Unknown; }
};

struct TestFallbackEnumParent {
  TestFallbackEnum uno;
};

struct TestFallbackEnumParentDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field<ttv::json::RequiredField>("uno", object.uno));
  }
};
}  // namespace

template <>
struct ttv::json::DefaultSchemaProvider<TestEnum> {
  using Type = EnumSchema<TestEnumDescription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<TestEnumParent> {
  using Type = ObjectSchema<TestEnumParentDescription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<TestFallbackEnum> {
  using Type = EnumSchema<TestFallbackEnumDescription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<TestFallbackEnumParent> {
  using Type = ObjectSchema<TestFallbackEnumParentDescription>;
};

TEST(JsonSerialization, ParseEnum) {
  ttv::json::Value value;
  value["uno"] = "first";
  value["dos"] = "second";
  value["tres"] = "third";

  TestEnumParent parent;
  ASSERT_TRUE(ttv::json::ToObject(value, parent));

  ASSERT_EQ(parent.uno, TestEnum::First);
  ASSERT_EQ(parent.dos, TestEnum::Second);
  ASSERT_EQ(parent.tres, TestEnum::Third);
}

TEST(JsonSerialization, ParseUnrecognizedEnumWithoutFallback) {
  ttv::json::Value value;
  value["uno"] = "first";
  value["dos"] = "second";
  value["tres"] = "umpteenth";

  TestEnumParent parent;
  ASSERT_FALSE(ttv::json::ToObject(value, parent));
}

TEST(JsonSerialization, ParseEnumWithFallback) {
  ttv::json::Value value;
  value["uno"] = "first";

  TestFallbackEnumParent parent;
  ASSERT_TRUE(ttv::json::ToObject(value, parent));

  ASSERT_EQ(parent.uno, TestFallbackEnum::First);
}

TEST(JsonSerialization, ParseUnrecognizedEnumWithFallback) {
  ttv::json::Value value;
  value["uno"] = "umpteenth";

  TestFallbackEnumParent parent;
  ASSERT_TRUE(ttv::json::ToObject(value, parent));

  ASSERT_EQ(parent.uno, TestFallbackEnum::Unknown);
}

TEST(JsonSerialization, EmitEnum) {
  TestEnumParent parent;
  parent.uno = TestEnum::First;
  parent.dos = TestEnum::Second;
  parent.tres = TestEnum::Third;

  ttv::json::Value value;
  ttv::json::ToJson(parent, value);

  ASSERT_EQ(value["uno"], "first");
  ASSERT_EQ(value["dos"], "second");
  ASSERT_EQ(value["tres"], "third");
}

namespace {
struct TestOptional {
  ttv::Optional<uint32_t> id;
  ttv::Optional<std::string> str;
  ttv::Optional<std::string> str2;
  std::string str3;
};

struct TestOptionalDescription {
  template <typename ObjectType>
  static auto BindFields(ObjectType& object) {
    return std::make_tuple(ttv::json::make_field<ttv::json::OptionalField>("id", object.id),
      ttv::json::make_field<ttv::json::RequiredField>("str", object.str),
      ttv::json::make_field<ttv::json::RequiredField>("str2", object.str2),
      ttv::json::make_field<ttv::json::OptionalField>("str3", object.str3));
  }
};
}  // namespace

template <>
struct ttv::json::DefaultSchemaProvider<TestOptional> {
  using Type = ObjectSchema<TestOptionalDescription>;
};

TEST(JsonSerialization, ParseOptionalObject_Success) {
  TestOptional output;

  bool success = ttv::json::ToObject(
    "{"
    "\"id\":9001,"
    "\"str\":\"test\","
    "\"str2\":null,"
    "\"str3\":\"\""
    "}",
    output);

  EXPECT_TRUE(success);
  EXPECT_EQ(*output.id, 9001);
  EXPECT_TRUE(output.str);
  EXPECT_FALSE(output.str2);
  EXPECT_TRUE(output.str3.empty());

  success = ttv::json::ToObject(
    "{"
    "\"id\":9001,"
    "\"str\":\"test\""
    "}",
    output);

  EXPECT_TRUE(success);
  EXPECT_EQ(*output.id, 9001);
  EXPECT_TRUE(output.str);
  EXPECT_FALSE(output.str2);
  EXPECT_TRUE(output.str3.empty());
}

TEST(JsonSerializationTest, ParseOptionalObject_Failure) {
  TestOptional output;

  bool success = ttv::json::ToObject(
    "{"
    "\"id\":9001,"
    "\"str\":\"test\","
    "\"str2\":10,"
    "\"str3\":\"\""
    "}",
    output);

  EXPECT_FALSE(success);
}
