/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/threadedeventscheduler.h"
#include "twitchsdk/core/timer.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class LambdaRetryTimerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();
    mEventScheduler = std::make_shared<ThreadedEventScheduler>("LambdaRetryTimerTest");
  }

  virtual void TearDownComponents() override {
    mEventScheduler->Shutdown([this] { mShutdown = true; });

    ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
    mEventScheduler.reset();
    SdkBaseTest::TearDownComponents();
  }

  std::shared_ptr<ThreadedEventScheduler> mEventScheduler;
  int32_t mTasksRun = 0;
  bool mShutdown = false;
};
}  // namespace

TEST_F(LambdaRetryTimerTest, SingleTests) {
  // Simple Start
  LambdaRetryTimer timer;
  timer.SetEventScheduler(mEventScheduler);
  timer.SetCallback([this]() { mTasksRun++; });
  timer.Start(100);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer.IsSet());
  ASSERT_TRUE(WaitUntilResult(150, [this]() { return mTasksRun == 1; }));
  ASSERT_FALSE(timer.IsSet());

  // Simple Start
  mTasksRun = 0;
  timer.Start(0);
  ASSERT_TRUE(WaitUntilResult(20, [this]() { return mTasksRun == 1; }));
  ASSERT_FALSE(timer.IsSet());

  // Simple Start + Subsequent Stop (to cancel it)
  mTasksRun = 0;
  timer.Start(100);
  ASSERT_EQ(mTasksRun, 0);
  ttv::Sleep(50);
  ASSERT_TRUE(timer.IsSet());
  timer.Stop();
  ASSERT_FALSE(WaitUntilResult(150, [this]() { return mTasksRun == 1; }));
  ASSERT_FALSE(timer.IsSet());
}

TEST_F(LambdaRetryTimerTest, LoopingTimer) {
  // Looping Timer Scenario
  int count = 0;
  LambdaRetryTimer timer;
  timer.SetEventScheduler(mEventScheduler);
  timer.SetCallback([this, &timer, &count]() {
    mTasksRun++;
    if (count < 3) {
      count++;
      timer.Start(100);
    }
  });
  timer.Start(50);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer.IsSet());

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mTasksRun == 1; }));
  ASSERT_TRUE(timer.IsSet());

  ASSERT_TRUE(WaitUntilResult(200, [this]() { return mTasksRun == 2; }));
  ASSERT_TRUE(timer.IsSet());

  ASSERT_TRUE(WaitUntilResult(200, [this]() { return mTasksRun == 3; }));

  ASSERT_TRUE(WaitUntilResult(200, [this]() { return mTasksRun == 4; }));

  ASSERT_FALSE(timer.IsSet());
}

TEST_F(LambdaRetryTimerTest, SortedOrder) {
  std::vector<uint64_t> tasksRun;

  std::vector<std::shared_ptr<LambdaRetryTimer>> timers;
  std::array<int, 13> arr{{160, 160, 40, 0, 40, 80, 0, 0, 160, 120, 120, 120, 40}};
  for (int i = 0; i < 13; i++) {
    std::shared_ptr<LambdaRetryTimer> timer = std::make_shared<LambdaRetryTimer>();
    timer->SetEventScheduler(mEventScheduler);
    timer->SetCallback([&tasksRun, i]() { tasksRun.emplace_back(i); });
    timer->Start(arr[i]);
    timers.emplace_back(timer);
  }

  ttv::Sleep(10);
  ASSERT_TRUE(timers[0]->IsSet());
  ASSERT_EQ(timers[0]->Stop(), TTV_EC_SUCCESS);
  ASSERT_FALSE(timers[0]->IsSet());
  ASSERT_TRUE(timers[1]->IsSet());
  ASSERT_TRUE(timers[2]->IsSet());
  ASSERT_FALSE(timers[3]->IsSet());
  ASSERT_EQ(timers[3]->Stop(), TTV_EC_OPERATION_FAILED);
  ASSERT_FALSE(timers[3]->IsSet());
  ASSERT_TRUE(timers[4]->IsSet());
  ASSERT_TRUE(timers[5]->IsSet());
  ASSERT_FALSE(timers[6]->IsSet());
  ASSERT_FALSE(timers[7]->IsSet());
  ASSERT_TRUE(timers[8]->IsSet());
  ASSERT_TRUE(timers[9]->IsSet());
  ASSERT_TRUE(timers[10]->IsSet());
  ASSERT_TRUE(timers[11]->IsSet());
  ASSERT_EQ(timers[11]->Stop(), TTV_EC_SUCCESS);
  ASSERT_FALSE(timers[11]->IsSet());
  ASSERT_TRUE(timers[12]->IsSet());

  ASSERT_TRUE(WaitUntilResult(300, [&tasksRun]() { return tasksRun.size() == 11; }));

  std::array<int, 11> testArr{{3, 6, 7, 2, 4, 12, 5, 9, 10, 1, 8}};

  for (int i = 0; i < 11; i++) {
    ASSERT_EQ(testArr[i], tasksRun[i]);
    ASSERT_FALSE(timers[i]->IsSet());
  }
}

TEST_F(LambdaRetryTimerTest, FailureCases) {
  LambdaRetryTimer timer;
  ASSERT_FALSE(timer.IsSet());
  ASSERT_EQ(timer.Stop(), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(timer.Start(100), TTV_EC_NOT_INITIALIZED);
  timer.SetEventScheduler(mEventScheduler);
  ASSERT_FALSE(timer.IsSet());
  ASSERT_EQ(timer.Stop(), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(timer.Start(100), TTV_EC_NOT_INITIALIZED);
  timer.SetCallback([this]() { mTasksRun++; });
  ASSERT_EQ(timer.Stop(), TTV_EC_OPERATION_FAILED);
  timer.Start(50);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer.IsSet());
  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mTasksRun == 1; }));
  ASSERT_FALSE(timer.IsSet());

  mTasksRun = 0;
  timer.Start(100);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer.IsSet());
  ttv::Sleep(50);
  timer.Start(100);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer.IsSet());

  ASSERT_FALSE(WaitUntilResult(200, [this]() { return mTasksRun == 2; }));
  ASSERT_EQ(mTasksRun, 1);
  ASSERT_FALSE(timer.IsSet());

  ASSERT_EQ(timer.Stop(), TTV_EC_OPERATION_FAILED);
}

TEST_F(LambdaRetryTimerTest, Shutdown) {
  std::shared_ptr<LambdaRetryTimer> timer = std::make_shared<LambdaRetryTimer>();
  timer->SetEventScheduler(mEventScheduler);
  timer->SetCallback([this]() { mTasksRun++; });
  timer->Start(100);
  ASSERT_TRUE(timer->IsSet());

  ttv::Sleep(50);
  ASSERT_EQ(mTasksRun, 0);
  ASSERT_TRUE(timer->IsSet());

  timer.reset();
  timer = nullptr;

  ASSERT_FALSE(WaitUntilResult(200, [this]() { return mTasksRun == 1; }));
}
