/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/optional.h"

#include "gtest/gtest.h"

TEST(OptionalTests, ValidValue) {
  ttv::Optional<int> optional(5);
  ASSERT_TRUE(optional);
  ASSERT_TRUE(optional.HasValue());
  ASSERT_EQ(optional, 5);

  int x = 1;
  optional = x;
  ASSERT_EQ(optional.Value(), x);

  ttv::Optional<int> optional2(x);
  ASSERT_EQ(*optional2, x);
}

TEST(OptionalTests, NoValueComparisons) {
  ttv::Optional<int> optional;
  ttv::Optional<int> optional2;

  ASSERT_FALSE(optional);
  ASSERT_FALSE(optional.HasValue());
  ASSERT_FALSE(optional2);
  ASSERT_FALSE(optional2.HasValue());

  ASSERT_EQ(optional, optional2);
  ASSERT_TRUE(optional == optional2);
  ASSERT_FALSE(optional != optional2);
  ASSERT_FALSE(optional < optional2);
  ASSERT_FALSE(optional > optional2);
  ASSERT_TRUE(optional <= optional2);
  ASSERT_TRUE(optional >= optional2);

  ASSERT_FALSE(optional == 0);
  ASSERT_TRUE(optional != 0);
  ASSERT_TRUE(optional < 0);
  ASSERT_FALSE(optional > 0);
  ASSERT_TRUE(optional <= 0);
  ASSERT_FALSE(optional >= 0);

  ASSERT_FALSE(0 == optional);
  ASSERT_TRUE(0 != optional);
  ASSERT_FALSE(0 < optional);
  ASSERT_TRUE(0 > optional);
  ASSERT_FALSE(0 <= optional);
  ASSERT_TRUE(0 >= optional);
}

TEST(OptionalTests, OptionalValuesComparisons) {
  ttv::Optional<int> optional(1);
  ttv::Optional<int> optional2;

  ASSERT_TRUE(optional);
  ASSERT_EQ(optional, 1);

  ASSERT_FALSE(optional == optional2);
  ASSERT_TRUE(optional != optional2);
  ASSERT_FALSE(optional < optional2);
  ASSERT_TRUE(optional > optional2);
  ASSERT_FALSE(optional <= optional2);
  ASSERT_TRUE(optional >= optional2);

  optional2 = optional;
  ASSERT_EQ(optional, optional2);
  ASSERT_TRUE(optional == optional2);
  ASSERT_FALSE(optional != optional2);
  ASSERT_FALSE(optional < optional2);
  ASSERT_FALSE(optional > optional2);
  ASSERT_TRUE(optional <= optional2);
  ASSERT_TRUE(optional >= optional2);

  *optional2 = 2;
  ASSERT_FALSE(optional == optional2);
  ASSERT_TRUE(optional != optional2);
  ASSERT_TRUE(optional < optional2);
  ASSERT_FALSE(optional > optional2);
  ASSERT_TRUE(optional <= optional2);
  ASSERT_FALSE(optional >= optional2);

  optional2 = std::move(optional);
  ASSERT_TRUE(optional2);
  ASSERT_EQ(optional2, 1);

  ttv::Optional<int> optional3(optional2);
  ASSERT_TRUE(optional3);
  ASSERT_EQ(optional3, optional2);
  ASSERT_EQ(optional3, 1);

  ttv::Optional<int> optional4(std::move(optional2));
  ASSERT_TRUE(optional4);
  ASSERT_EQ(optional4, optional2);
  ASSERT_EQ(optional4, 1);
}

TEST(OptionalTests, ValueComparisons) {
  ttv::Optional<int> optional(1);

  const ttv::Optional<int>& ref1 = optional;
  const int& ref2 = 1;

  ASSERT_TRUE(ref1 == ref2);

  ASSERT_EQ(optional, 1);
  ASSERT_TRUE(optional == 1);
  ASSERT_TRUE(optional != 2);
  ASSERT_TRUE(optional < 2);
  ASSERT_TRUE(optional > 0);
  ASSERT_TRUE(optional <= 2);
  ASSERT_TRUE(optional <= 1);
  ASSERT_TRUE(optional >= 0);
  ASSERT_TRUE(optional >= 1);

  ASSERT_FALSE(optional == 2);
  ASSERT_FALSE(optional != 1);
  ASSERT_FALSE(optional < 0);
  ASSERT_FALSE(optional > 2);
  ASSERT_FALSE(optional <= 0);
  ASSERT_FALSE(optional >= 2);

  ASSERT_EQ(1, optional);
  ASSERT_TRUE(1 == optional);
  ASSERT_TRUE(2 != optional);
  ASSERT_TRUE(2 > optional);
  ASSERT_TRUE(0 < optional);
  ASSERT_TRUE(2 >= optional);
  ASSERT_TRUE(1 >= optional);
  ASSERT_TRUE(0 <= optional);
  ASSERT_TRUE(1 <= optional);

  ASSERT_FALSE(2 == optional);
  ASSERT_FALSE(1 != optional);
  ASSERT_FALSE(0 > optional);
  ASSERT_FALSE(2 < optional);
  ASSERT_FALSE(0 >= optional);
  ASSERT_FALSE(2 <= optional);
}
