/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "twitchsdk/core/stringutilities.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::test;

/**
 * Tests for ParseNum overloads in stringutilities.h
 * http://en.cppreference.com/w/c/io/fscanf
 * http://www.cplusplus.com/reference/cstdio/scanf/
 * https://en.wikipedia.org/wiki/C_data_types
 */
TEST_F(SdkBaseTest, ParseSignedNum) {
  // int - at least [-32767, 32767]
  std::string int1 = "32767";
  std::string int2 = "-32767";
  int intOut;
  ASSERT_TRUE(ParseNum(int1, intOut));
  ASSERT_EQ(intOut, 32767);
  ASSERT_TRUE(ParseNum(int2, intOut));
  ASSERT_EQ(intOut, -32767);

  // short int - at least [-32767, 32767]
  short int shortIntOut;
  ASSERT_TRUE(ParseNum(int1, shortIntOut));
  ASSERT_EQ(shortIntOut, 32767);
  ASSERT_TRUE(ParseNum(int2, shortIntOut));
  ASSERT_EQ(shortIntOut, -32767);

  // signed char - at least [-127, 127]
  std::string signedChar1 = "127";
  std::string signedChar2 = "-127";
  signed char signedCharOut;
  ASSERT_TRUE(ParseNum(signedChar1, signedCharOut));
  ASSERT_EQ(signedCharOut, 127);
  ASSERT_TRUE(ParseNum(signedChar2, signedCharOut));
  ASSERT_EQ(signedCharOut, -127);

  // long int - at least [-2147483647, 2147483647]
  std::string longInt1 = "2147483647";
  std::string longInt2 = "-2147483647";
  long int longIntOut;
  ASSERT_TRUE(ParseNum(longInt1, longIntOut));
  ASSERT_EQ(longIntOut, 2147483647);
  ASSERT_TRUE(ParseNum(longInt2, longIntOut));
  ASSERT_EQ(longIntOut, -2147483647);

  // long long int - at least [-9223372036854775807, 9223372036854775807]
  std::string longLongInt1 = "9223372036854775807";
  std::string longLongInt2 = "-9223372036854775807";
  long long int longLongIntOut;
  ASSERT_TRUE(ParseNum(longLongInt1, longLongIntOut));
  ASSERT_EQ(longLongIntOut, 9223372036854775807);
  ASSERT_TRUE(ParseNum(longLongInt2, longLongIntOut));
  ASSERT_EQ(longLongIntOut, -9223372036854775807);
}

TEST_F(SdkBaseTest, ParseUnsignedNum) {
  std::string zero = "0";

  // uint16_t - [0, 65535]
  std::string uint16Int = "65535";
  uint16_t uint16IntOut;
  ASSERT_TRUE(ParseNum(zero, uint16IntOut));
  ASSERT_EQ(uint16IntOut, 0);
  ASSERT_TRUE(ParseNum(uint16Int, uint16IntOut));
  ASSERT_EQ(uint16IntOut, 65535);
  ASSERT_TRUE(ParseNum("-1", uint16IntOut));
  ASSERT_TRUE(uint16IntOut > 0 && uint16IntOut == 65535);

  // uint32_t - [0, 4294967295]
  std::string uint32Int = "4294967295";
  uint32_t uint32IntOut;
  ASSERT_TRUE(ParseNum(zero, uint32IntOut));
  ASSERT_EQ(uint32IntOut, 0);
  ASSERT_TRUE(ParseNum(uint32Int, uint32IntOut));
  ASSERT_EQ(uint32IntOut, 4294967295);
  ASSERT_TRUE(ParseNum("-1", uint32IntOut));
  ASSERT_TRUE(uint32IntOut > 0 && uint32IntOut == 4294967295);

  // uint64_t - [0, 18446744073709551615]
  std::string uint64Int = "18446744073709551615";
  uint64_t uint64IntOut;
  ASSERT_TRUE(ParseNum(zero, uint64IntOut));
  ASSERT_EQ(uint64IntOut, 0);
  ASSERT_TRUE(ParseNum(uint64Int, uint64IntOut));
  ASSERT_EQ(uint64IntOut, 18446744073709551615U);
  ASSERT_TRUE(ParseNum("-1", uint32IntOut));
  ASSERT_TRUE(uint64IntOut > 0 && uint64IntOut == 18446744073709551615U);

  // char - could be signed or unsigned - at least [0, 127]
  std::string charInt1 = " ";
  std::string charInt2 = "9";
  std::string charInt3 = "A";
  std::string charInt4 = "~";
  char charOut;
  ASSERT_TRUE(ParseNum(charInt1, charOut));
  ASSERT_EQ(charOut, 32);
  ASSERT_TRUE(ParseNum(charInt2, charOut));
  ASSERT_EQ(charOut, 57);
  ASSERT_TRUE(ParseNum(charInt3, charOut));
  ASSERT_EQ(charOut, 65);
  ASSERT_TRUE(ParseNum(charInt4, charOut));
  ASSERT_EQ(charOut, 126);

  // unsigned char - at least [0, 255]
  std::string unsignedChar = "255";
  unsigned char unsignedCharOut;
  ASSERT_TRUE(ParseNum(zero, unsignedCharOut));
  ASSERT_EQ(unsignedCharOut, 0);
  ASSERT_TRUE(ParseNum(unsignedChar, unsignedCharOut));
  ASSERT_EQ(unsignedCharOut, 255);
  ASSERT_TRUE(ParseNum("-1", unsignedCharOut));
  ASSERT_TRUE(unsignedCharOut > 0);

  // unsigned short int - at least [0, 65535]
  std::string unsignedShortInt = "65535";
  unsigned short int unsignedShortIntOut;
  ASSERT_TRUE(ParseNum(zero, unsignedShortIntOut));
  ASSERT_EQ(unsignedShortIntOut, 0);
  ASSERT_TRUE(ParseNum(unsignedShortInt, unsignedShortIntOut));
  ASSERT_EQ(unsignedShortIntOut, 65535);
  ASSERT_TRUE(ParseNum("-1", unsignedShortIntOut));
  ASSERT_TRUE(unsignedShortIntOut > 0);

  // unsigned int - at least [0, 65535]
  std::string unsignedInt = "65535";
  unsigned int unsignedIntOut;
  ASSERT_TRUE(ParseNum(zero, unsignedIntOut));
  ASSERT_EQ(unsignedIntOut, 0);
  ASSERT_TRUE(ParseNum(unsignedInt, unsignedIntOut));
  ASSERT_EQ(unsignedIntOut, 65535);
  ASSERT_TRUE(ParseNum("-1", unsignedIntOut));
  ASSERT_TRUE(unsignedIntOut > 0);

  // unsigned long int - at least [0, 4294967295]
  std::string unsignedLongInt = "4294967295";
  unsigned long int unsignedLongIntOut;
  ASSERT_TRUE(ParseNum(zero, unsignedLongIntOut));
  ASSERT_EQ(unsignedLongIntOut, 0);
  ASSERT_TRUE(ParseNum(unsignedLongInt, unsignedLongIntOut));
  ASSERT_EQ(unsignedLongIntOut, 4294967295U);
  ASSERT_TRUE(ParseNum("-1", unsignedLongIntOut));
  ASSERT_TRUE(unsignedLongIntOut > 0);

  // unsigned long long int - at least [0, 18446744073709551615]
  std::string unsignedLongLongInt = "18446744073709551615";
  unsigned long long int unsignedLongLongIntOut;
  ASSERT_TRUE(ParseNum(zero, unsignedLongLongIntOut));
  ASSERT_EQ(unsignedLongLongIntOut, 0);
  ASSERT_TRUE(ParseNum(unsignedLongLongInt, unsignedLongLongIntOut));
  ASSERT_EQ(unsignedLongLongIntOut, 18446744073709551615U);
  ASSERT_TRUE(ParseNum("-1", unsignedLongLongIntOut));
  ASSERT_TRUE(unsignedLongLongIntOut > 0);
}

TEST_F(SdkBaseTest, ParseFloatNum) {
  std::string zero = "0.0";
  std::string float1 = "100.0";
  std::string float2 = "0.125";
  std::string double1 = "16777217";              // not representable by a float (2^24 + 1) - 23 mantissa bits
  std::string longDouble1 = "9007199254740993";  // not representable by a double (2^53 + 1) - 52 mantissa bits

  float floatOut;
  ASSERT_TRUE(ParseNum(zero, floatOut));
  ASSERT_FLOAT_EQ(floatOut, 0.0);
  ASSERT_TRUE(ParseNum(float1, floatOut));
  ASSERT_FLOAT_EQ(floatOut, 100.0);
  ASSERT_TRUE(ParseNum(float2, floatOut));
  ASSERT_FLOAT_EQ(floatOut, 0.125);
  ASSERT_TRUE(ParseNum(double1, floatOut));
  ASSERT_FLOAT_EQ(floatOut, 16777216.0);

  double doubleOut;
  ASSERT_TRUE(ParseNum(zero, doubleOut));
  ASSERT_DOUBLE_EQ(doubleOut, 0.0);
  ASSERT_TRUE(ParseNum(float1, doubleOut));
  ASSERT_DOUBLE_EQ(doubleOut, 100.0);
  ASSERT_TRUE(ParseNum(float2, doubleOut));
  ASSERT_DOUBLE_EQ(doubleOut, 0.125);
  ASSERT_TRUE(ParseNum(double1, doubleOut));
  ASSERT_DOUBLE_EQ(doubleOut, 16777217.0);
  ASSERT_TRUE(ParseNum(longDouble1, doubleOut));
  ASSERT_DOUBLE_EQ(doubleOut, 9007199254740992.0);

  // on Windows, long double is a synonym for double, other compilers use 80-bit extended precision
  long double longDoubleOut;
  ASSERT_TRUE(ParseNum(zero, longDoubleOut));
  ASSERT_DOUBLE_EQ(longDoubleOut, 0.0);
  ASSERT_TRUE(ParseNum(float1, longDoubleOut));
  ASSERT_DOUBLE_EQ(longDoubleOut, 100.0);
  ASSERT_TRUE(ParseNum(float2, longDoubleOut));
  ASSERT_DOUBLE_EQ(longDoubleOut, 0.125);
  ASSERT_TRUE(ParseNum(double1, longDoubleOut));
  ASSERT_DOUBLE_EQ(longDoubleOut, 16777217.0);
  ASSERT_TRUE(ParseNum(longDouble1, longDoubleOut));

#if TTV_PLATFORM_WIN32
  ASSERT_DOUBLE_EQ(longDoubleOut, 9007199254740992.0);
#else
  ASSERT_DOUBLE_EQ(longDoubleOut, 9007199254740993.0L);
#endif
}
