/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/pollingeventscheduler.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class PollingEventSchedulerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();
    mEventScheduler = std::make_shared<PollingEventScheduler>("PollingEventSchedulerTest");
  }

  virtual void TearDownComponents() override {
    mEventScheduler.reset();
    SdkBaseTest::TearDownComponents();
  }

  std::shared_ptr<PollingEventScheduler> mEventScheduler;
  int32_t mTasksRun = 0;
  bool mShutdown = false;
};
}  // namespace

TEST_F(PollingEventSchedulerTest, SimpleInsertsNoDelay) {
  TTV_ErrorCode ec;
  for (int i = 0; i < 100; i++) {
    Result<TaskId> result = mEventScheduler->ScheduleTask({[this]() { mTasksRun++; }});
    ASSERT_FALSE(result.IsError());
  }

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 100);
  mTasksRun = 0;

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 0);
  mTasksRun = 0;

  ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);
  ASSERT_FALSE(mShutdown);

  mEventScheduler->Update();
  ASSERT_TRUE(mShutdown);
}

TEST_F(PollingEventSchedulerTest, StressCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 1000});
        ASSERT_FALSE(result.IsError());
        ASSERT_EQ(mEventScheduler->CancelTask(result.GetResult()), TTV_EC_SUCCESS);
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 0);

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);
  mEventScheduler->Update();
  ASSERT_TRUE(mShutdown);
}

TEST_F(PollingEventSchedulerTest, StressHalfCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 500});
        ASSERT_FALSE(result.IsError());

        TaskId id = result.GetResult();
        if (id % 2 == 0) {
          ASSERT_EQ(mEventScheduler->CancelTask(id), TTV_EC_SUCCESS);
        }
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 0);

  ttv::Sleep(1000);  // Wait for the delay to pass for the scheduled tasks

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, (numThreads * tasksPerThread / 2));

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);
  mEventScheduler->Update();
  ASSERT_TRUE(mShutdown);
}

TEST_F(PollingEventSchedulerTest, SortedOrder) {
  // Test the sorted order of the output (with some cancellation)
  std::vector<TaskId> tasksRun;

  Result<TaskId> result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(1); }, 300});
  ASSERT_TRUE(result.GetResult() == 1);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(2); }, 300});
  ASSERT_TRUE(result.GetResult() == 2);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(3); }, 75});
  ASSERT_TRUE(result.GetResult() == 3);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(4); }, 0});
  ASSERT_TRUE(result.GetResult() == 4);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(5); }, 75});
  ASSERT_TRUE(result.GetResult() == 5);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(6); }, 300});
  ASSERT_TRUE(result.GetResult() == 6);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(7); }, 0});
  ASSERT_TRUE(result.GetResult() == 7);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(8); }, 0});
  ASSERT_TRUE(result.GetResult() == 8);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(9); }, 300});
  ASSERT_TRUE(result.GetResult() == 9);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(10); }, 300});
  ASSERT_TRUE(result.GetResult() == 10);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(11); }, 175});
  ASSERT_TRUE(result.GetResult() == 11);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(12); }, 300});
  ASSERT_TRUE(result.GetResult() == 12);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(13); }, 75});
  ASSERT_TRUE(result.GetResult() == 13);

  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_SUCCESS);
  ASSERT_EQ(mEventScheduler->CancelTask(5), TTV_EC_SUCCESS);
  ASSERT_EQ(mEventScheduler->CancelTask(11), TTV_EC_SUCCESS);

  ASSERT_EQ(tasksRun.size(), 0);

  mEventScheduler->Update();
  ASSERT_EQ(tasksRun.size(), 3);

  ttv::Sleep(100);
  mEventScheduler->Update();
  ASSERT_EQ(tasksRun.size(), 5);

  ttv::Sleep(300);
  mEventScheduler->Update();
  ASSERT_EQ(tasksRun.size(), 10);

  mEventScheduler->Update();
  ASSERT_EQ(tasksRun.size(), 10);

  std::array<int, 10> arr{{4, 7, 8, 3, 13, 2, 6, 9, 10, 12}};

  for (int i = 0; i < 10; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);
  mEventScheduler->Update();
  ASSERT_TRUE(mShutdown);
}

TEST_F(PollingEventSchedulerTest, FailureCases) {
  // Test various failure cases

  ASSERT_EQ(mEventScheduler->CancelTask(0), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(100), TTV_EC_OPERATION_FAILED);

  Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 0});
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 1);

  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 1);

  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_OPERATION_FAILED);

  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 100000});
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 2);
  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 1);

  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_SUCCESS);
  mEventScheduler->Update();
  ASSERT_EQ(mTasksRun, 1);

  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mTasksRun, 1);

  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 100000});
  mEventScheduler->Update();
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 3);
  ASSERT_EQ(mTasksRun, 1);

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);
  mEventScheduler->Update();
  ASSERT_TRUE(mShutdown);
  ASSERT_EQ(mTasksRun, 1);

  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 1000});
  ASSERT_TRUE(result.IsError());
  ASSERT_EQ(result.GetErrorCode(), TTV_EC_NOT_INITIALIZED);
  mEventScheduler->Update();

  ASSERT_EQ(mEventScheduler->CancelTask(3), TTV_EC_NOT_INITIALIZED);

  ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_NOT_INITIALIZED);

  ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_NOT_INITIALIZED);
}
