
/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2015-2017 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/channel/channelstatus.h"
#include "twitchsdk/core/channel/ichannellistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class TestChannelListener : public IChannelListener {
 public:
  using ProfileImageUpdatedCallback = std::function<void(const std::vector<ProfileImage>& images)>;

  virtual void StreamUp(uint32_t /*playDelaySeconds*/) override {}

  virtual void StreamDown() override {}

  virtual void StreamViewerCountChanged(uint32_t /*viewerCount*/) override {}

  virtual void StreamTriggeredMidroll(uint32_t /*durationSeconds*/) override {}

  virtual void StreamReceivedWatchPartyUpdate(const WatchPartyUpdate& /*update*/) override {}

  virtual void ProfileImageUpdated(const std::vector<ProfileImage>& images) override {
    if (profileImageUpdatedCallback != nullptr) {
      profileImageUpdatedCallback(images);
    }
  }

  virtual void StreamInfoUpdated(StreamInfoUpdate&& /*info*/) override {}

  virtual void SquadUpdated(SquadInfo&& /*info*/) override {}

  virtual void SquadLeft() override {}

  virtual void PixelTrackingUpdate(bool /*refresh*/) override {}

  ProfileImageUpdatedCallback profileImageUpdatedCallback;
};

class ProfileUploadTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    InitializeModule(mCoreApi);
    AddModule(mCoreApi);

    mUserInfo.userId = 9001;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mHttpRequest->AddResponse("https://api.twitch.tv/kraken/users/9001/upload_image")
      .SetResponseBodyFromFile("core/getprofileimageurltask.json")
      .SetType(HttpRequestType::HTTP_POST_REQUEST)
      .Done();

    mHttpRequest
      ->AddResponse(
        "https://prod-web-upload-service-ingest.s3-accelerate.amazonaws.com/f61b6216-f2a6-45ca-a8d5-8f5613d88093?X-Amz-Algorithm=AWS4-HMAC-SHA256\u0026X-Amz-Credential=ASIAJ4MSIQNVTARCK7HQ%2F20170911%2Fus-west-2%2Fs3%2Faws4_request\u0026X-Amz-Date=20170911T200456Z\u0026X-Amz-Expires=900\u0026X-Amz-Security-Token=FQoDYXdzENz%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDP91q2ARZTdu%2FLqL3CK3A1t7gyqQgBtoTLlRYVkjS63oowXyFcsUVdc9k%2BbD7fjLSYBzlL1rT1yOCdtkf6ICvzdFx94ptLUDm1EYdb2v5gS9m23O%2B%2BzrR7JWMJ6r2ikWkmSc2%2BWn52nV7XXB2nfE3pd4KaUiuTZnooH7pjWY0i%2Btygruejzs5Qe4YyVwo9fhP8rytI%2FTaeCACW8gtWZtyc3ENIEP9xjXReMgoEb%2B%2BISegP42M%2FasdGN5aIpN5mbO31K1lkOef4kin8y0x3tYzDOkLkblwRkIUiUXEM%2FbN3PLsY94ndqgz1%2FkANZ81cfXCSL%2BDngne3tXBAyNrZGN%2Fmgdtt5dv2hfROgRfesI2lViJ3Ccr9N%2FDi%2FDHw8GEna%2FJUqU%2B%2B%2FxF%2B%2FyQtNPfjt4cy2l1a9Yowt6Z2b7lhFJtsUK5AmN4GEaSacBJscTSeB4%2B5hmH2AgCYiWnlhFfxzb33tpFxdic2Jn%2F2IzTE6UCl6cf7rR9uHkQXPMPMBEMqmLMyaGH7gIQ1B9vPicPeyq%2BjRyb2gS6dobjIIoHH8M%2BBYxO1b01yPAkEga3k0DaAE%2FLnxu8RIX5mUTQLwICKYNQi0R7Zivccoonr%2FbzQU%3D\u0026X-Amz-SignedHeaders=host\u0026X-Amz-Signature=c0db691066213655041616f67276e405cd77eb9f659aeddfecbefbad54e5205f")
      .SetResponseBody("")
      .SetStatusCode(200)
      .SetType(HttpRequestType::HTTP_PUT_REQUEST)
      .Done();

    mChannelListener = std::make_shared<TestChannelListener>();
  }

  std::shared_ptr<CoreAPI> mCoreApi;

 protected:
  std::shared_ptr<TestChannelListener> mChannelListener;
  UserInfo mUserInfo;
};
}  // namespace

TEST_F(ProfileUploadTest, UploadSuccess) {
  int receivedCallbacks = 0;
  mChannelListener->profileImageUpdatedCallback = [&receivedCallbacks](const std::vector<ProfileImage>& images) {
    ASSERT_EQ(images.size(), 2);
    ASSERT_EQ(images[0].width, 150);
    ASSERT_EQ(images[0].height, 150);
    ASSERT_EQ(images[0].url, "https://150x150.png");
    ASSERT_EQ(images[0].format, "png");
    ASSERT_EQ(images[1].width, 28);
    ASSERT_EQ(images[1].height, 28);
    ASSERT_EQ(images[1].url, "https://28x28.png");
    ASSERT_EQ(images[1].format, "png");
    receivedCallbacks++;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(9001, 9001, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  std::string image = "An image file to be uploaded.";
  ec = channelStatus->UploadProfileImage(
    image.c_str(), image.size(), [&receivedCallbacks](TTV_ErrorCode ec, const std::vector<ProfileImage>& images) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(images.size(), 2);
      ASSERT_EQ(images[0].width, 150);
      ASSERT_EQ(images[0].height, 150);
      ASSERT_EQ(images[0].url, "https://150x150.png");
      ASSERT_EQ(images[0].format, "png");
      ASSERT_EQ(images[1].width, 28);
      ASSERT_EQ(images[1].height, 28);
      ASSERT_EQ(images[1].url, "https://28x28.png");
      ASSERT_EQ(images[1].format, "png");
      receivedCallbacks++;
    });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("user-image-update.9001"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("user-image-update.9001",
    "{\"user_id\":\"9001\",\"type\":\"user_image_update\",\"image_type\":\"profile_image\",\"new_image\":{\"150x150\":{\"width\":150,\"height\":150,\"url\":\"https://150x150.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"150x150\",\"format\":\"png\",\"new_format\":true},\"28x28\":{\"width\":28,\"height\":28,\"url\":\"https://28x28.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"28x28\",\"format\":\"png\",\"new_format\":true}},\"status\":\"SUCCESS\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}");

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [&receivedCallbacks]() { return receivedCallbacks == 2; }, GetDefaultUpdateFunc()));
}

TEST_F(ProfileUploadTest, UploadSuccessWithoutListener) {
  int receivedCallbacks = 0;

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(9001, 9001, nullptr, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  std::string image = "An image file to be uploaded.";
  ec = channelStatus->UploadProfileImage(
    image.c_str(), image.size(), [&receivedCallbacks](TTV_ErrorCode ec, const std::vector<ProfileImage>& images) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(images.size(), 2);
      ASSERT_EQ(images[0].width, 150);
      ASSERT_EQ(images[0].height, 150);
      ASSERT_EQ(images[0].url, "https://150x150.png");
      ASSERT_EQ(images[0].format, "png");
      ASSERT_EQ(images[1].width, 28);
      ASSERT_EQ(images[1].height, 28);
      ASSERT_EQ(images[1].url, "https://28x28.png");
      ASSERT_EQ(images[1].format, "png");
      receivedCallbacks++;
    });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("user-image-update.9001"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("user-image-update.9001",
    "{\"user_id\":\"9001\",\"type\":\"user_image_update\",\"image_type\":\"profile_image\",\"new_image\":{\"150x150\":{\"width\":150,\"height\":150,\"url\":\"https://150x150.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"150x150\",\"format\":\"png\",\"new_format\":true},\"28x28\":{\"width\":28,\"height\":28,\"url\":\"https://28x28.png\",\"uid\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\",\"size\":\"28x28\",\"format\":\"png\",\"new_format\":true}},\"status\":\"SUCCESS\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}");

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [&receivedCallbacks]() { return receivedCallbacks == 1; }, GetDefaultUpdateFunc()));
}

TEST_F(ProfileUploadTest, UploadFailure_ImageValidation) {
  bool receivedCallback = false;
  bool listenerCallback = false;

  mChannelListener->profileImageUpdatedCallback = [&listenerCallback](const std::vector<ProfileImage>& /*images*/) {
    listenerCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(9001, 9001, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  std::string image = "An file that isn't an image.";
  ec = channelStatus->UploadProfileImage(
    image.c_str(), image.size(), [&receivedCallback](TTV_ErrorCode ec, const std::vector<ProfileImage>& /*images*/) {
      ASSERT_TRUE(TTV_FAILED(ec));
      ASSERT_EQ(ec, TTV_EC_PROFILEIMAGE_IMAGE_VALIDATION_FAILED);
      receivedCallback = true;
    });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("user-image-update.9001"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("user-image-update.9001",
    "{\"user_id\":\"9001\",\"type\":\"user_image_update\",\"image_type\":\"profile_image\",\"new_image\":null,\"status\":\"IS_IMAGE_VALIDATION_FAILED\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  ASSERT_FALSE(listenerCallback);
}

TEST_F(ProfileUploadTest, UploadFailure_SizeValidation) {
  bool receivedCallback = false;
  bool listenerCallback = false;

  mChannelListener->profileImageUpdatedCallback = [&listenerCallback](const std::vector<ProfileImage>& /*images*/) {
    listenerCallback = true;
  };

  std::shared_ptr<IChannelStatus> channelStatus;
  TTV_ErrorCode ec = mCoreApi->CreateChannelStatus(9001, 9001, mChannelListener, channelStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  std::string image = "An image that is too big.";
  ec = channelStatus->UploadProfileImage(
    image.c_str(), image.size(), [&receivedCallback](TTV_ErrorCode ec, const std::vector<ProfileImage>& /*images*/) {
      ASSERT_TRUE(TTV_FAILED(ec));
      ASSERT_EQ(ec, TTV_EC_PROFILEIMAGE_SIZE_VALIDATION_FAILED);
      receivedCallback = true;
    });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("user-image-update.9001"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("user-image-update.9001",
    "{\"user_id\":\"9001\",\"type\":\"user_image_update\",\"image_type\":\"profile_image\",\"new_image\":null,\"status\":\"FILE_SIZE_VALIDATION_FAILED\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  ASSERT_FALSE(listenerCallback);
}
