/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

#include <thread>

#include "gtest/gtest.h"

using namespace ttv::test;

namespace {
using namespace ttv;

class PubSubTestListener : public PubSubClient::IListener {
  virtual void OnStateChanged(PubSubClient* /*source*/, PubSubState /*state*/, TTV_ErrorCode /*ec*/) {}
};

class PubSubTestTopicListener : public PubSubClient::ITopicListener {
  virtual void OnTopicSubscribeStateChanged(PubSubClient* /*source*/, const std::string& /*topic*/,
    PubSubClient::SubscribeState::Enum /*state*/, TTV_ErrorCode /*ec*/) {}
  virtual void OnTopicMessageReceived(
    PubSubClient* /*source*/, const std::string& /*topic*/, const json::Value& /*msg*/) {}
  virtual void OnTopicListenerRemoved(PubSubClient* /*source*/, const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {}
};

class TestUserListener : public IUserListener {
 public:
  TestUserListener() : mReceivedAuthError(false) {}
  bool GetReceivedAuthError() { return mReceivedAuthError; }
  virtual void OnUserLogInComplete(User* /*source*/, TTV_ErrorCode /*ec*/) {}
  virtual void OnUserLogOutComplete(User* /*source*/, TTV_ErrorCode /*ec*/) {}
  virtual void OnUserInfoFetchComplete(User* /*source*/, TTV_ErrorCode /*ec*/) {}
  virtual void OnUserAuthenticationIssue(
    User* /*source*/, std::shared_ptr<const OAuthToken> /*oauthToken*/, TTV_ErrorCode /*ec*/) {
    mReceivedAuthError = true;
  }

 private:
  bool mReceivedAuthError;
};
}  // namespace

TEST_F(SdkBaseTest, PubSubClientTest) {
  std::shared_ptr<TaskRunner> taskRunner = CreateTaskRunner();
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);

  userRepository->SetTaskRunner(taskRunner);
  InitializeComponent(userRepository);

  auto user = userRepository->RegisterUser(9001);
  user->SetOAuthToken(std::make_shared<OAuthToken>("auth_token"));

  std::shared_ptr<PubSubClient> pubsub = std::make_shared<PubSubClient>(user, nullptr);
  AddComponent(pubsub);
  pubsub->SetTaskRunner(taskRunner);

  std::shared_ptr<PubSubTestListener> listener = std::make_shared<PubSubTestListener>();
  pubsub->Initialize();

  pubsub->AddListener(listener);

  // Wait for connection
  pubsub->Connect();
  std::function<bool()> checkFunc = [pubsub]() { return pubsub->GetConnectionState() == PubSubState::Connected; };
  ASSERT_TRUE(WaitUntilResult(3000, checkFunc));

  auto userListener = std::make_shared<TestUserListener>();
  user->AddListener(userListener);

  mPubSubTestUtility.SetListenErrorString("ERR_BADAUTH");

  pubsub->AddTopicListener("test_topic", std::make_shared<PubSubTestTopicListener>());

  ASSERT_TRUE(WaitUntilResult(1000, [userListener]() { return userListener->GetReceivedAuthError(); }));

  pubsub->Disconnect();
}
