/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "pubsubtestutility.h"

#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/user/user.h"

#include "gtest/gtest.h"

ttv::test::PubSubTestUtility::PubSubTestUtility()
    : mTestWebSocketFactory(std::make_shared<TestWebSocketFactory>()),
      mTestWebSocket(std::make_shared<TestWebSocket>()) {
  CreateMutex(mMutex, "PubSubTestUtility");
}

void ttv::test::PubSubTestUtility::SetUpStubs() {
  SetUpStubs("wss://pubsub-edge.twitch.tv");
}

void ttv::test::PubSubTestUtility::SetUpStubs(const std::string& uri) {
  mTestWebSocket->SetSendCallback([this](const std::string& sentPayload) { ProcessSentPayload(sentPayload); });

  mTestWebSocketFactory->SetSocket(mTestWebSocket, uri);
  RegisterWebSocketFactory(mTestWebSocketFactory);
}

void ttv::test::PubSubTestUtility::SetUpComponents(std::shared_ptr<TaskRunner> taskRunner, std::shared_ptr<User> user) {
  mPubSubClient = std::make_shared<PubSubClient>(user, nullptr);
  mPubSubClient->SetTaskRunner(taskRunner);
  user->GetComponentContainer()->SetComponent(ttv::PubSubClient::GetComponentName(), mPubSubClient);
}

void ttv::test::PubSubTestUtility::TearDownComponents() {}

void ttv::test::PubSubTestUtility::TearDownStubs() {
  UnregisterWebSocketFactory(mTestWebSocketFactory);
}

bool ttv::test::PubSubTestUtility::IsSubscribedToTopic(const std::string& topic) const {
  ttv::AutoMutex lock(mMutex.get());
  return std::find(mSubscribedTopics.begin(), mSubscribedTopics.end(), topic) != mSubscribedTopics.end();
}

void ttv::test::PubSubTestUtility::PushPubSubMessage(const std::string& topic, const std::string& message) {
  json::Value pubSubWrapper;
  pubSubWrapper["type"] = "MESSAGE";

  json::Value data;
  data["topic"] = topic;
  data["message"] = message;

  pubSubWrapper["data"] = std::move(data);

  json::FastWriter writer;
  mTestWebSocket->PushReceivedPayload(writer.write(pubSubWrapper), nullptr);
}

void ttv::test::PubSubTestUtility::PushPubSubMessage(const std::string& topic, const ttv::json::Value& message) {
  json::FastWriter writer;
  PushPubSubMessage(topic, writer.write(message));
}

void ttv::test::PubSubTestUtility::ProcessSentPayload(const std::string& sentPayload) {
  json::Reader reader;
  json::Value rootValue;
  ASSERT_TRUE(reader.parse(sentPayload, rootValue));

  const auto& sentData = rootValue["data"];
  ASSERT_TRUE(sentData.isObject());

  const auto& topics = sentData["topics"];
  ASSERT_TRUE(topics.isArray());

  {
    ttv::AutoMutex lock(mMutex.get());
    for (const auto& topic : topics) {
      ASSERT_TRUE(topic.isString());
      mSubscribedTopics.push_back(topic.asString());
    }
  }

  json::Value response;
  response["type"] = "RESPONSE";
  response["nonce"] = rootValue["nonce"];

  if (!mListenErrorString.empty()) {
    response["error"] = mListenErrorString;
  }

  json::FastWriter writer;
  mTestWebSocket->PushReceivedPayload(writer.write(response), nullptr);
}
