/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/result.h"

#include <type_traits>

#include <memory>

#include "gtest/gtest.h"

namespace {
template <typename ValueType>
void TestCreateSuccessResult(ValueType&& value) {
  auto copiedValue = value;
  auto result = ttv::MakeSuccessResult(std::forward<ValueType>(value));
  static_assert(std::is_same<decltype(result), ttv::Result<std::decay_t<ValueType>>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult(), copiedValue);
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);
}
}  // namespace

TEST(ResultTests, TestCreateSuccessResults) {
  TestCreateSuccessResult(1);
  TestCreateSuccessResult("hello");
  TestCreateSuccessResult(std::string{"hello"});
  TestCreateSuccessResult(std::vector<uint32_t>({2, 3, 4, 5, 6}));
}

TEST(ResultTests, TestCopyErrorResult) {
  auto errorResult = ttv::MakeErrorResult(TTV_EC_UNKNOWN_ERROR);
  static_assert(std::is_same<decltype(errorResult), ttv::Result<void>>::value, "Unexpected result type.");

  EXPECT_FALSE(errorResult.IsSuccess());
  EXPECT_TRUE(errorResult.IsError());
  EXPECT_EQ(errorResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);

  ttv::Result<void> copyResult{errorResult};

  EXPECT_FALSE(copyResult.IsSuccess());
  EXPECT_TRUE(copyResult.IsError());
  EXPECT_EQ(copyResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);
}

TEST(ResultTests, TestCopyAssignErrorResult) {
  auto errorResult = ttv::MakeErrorResult(TTV_EC_UNKNOWN_ERROR);
  static_assert(std::is_same<decltype(errorResult), ttv::Result<void>>::value, "Unexpected result type.");

  EXPECT_FALSE(errorResult.IsSuccess());
  EXPECT_TRUE(errorResult.IsError());
  EXPECT_EQ(errorResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);

  auto copyResult = ttv::MakeErrorResult(TTV_EC_UNIMPLEMENTED);
  EXPECT_FALSE(copyResult.IsSuccess());
  EXPECT_TRUE(copyResult.IsError());
  EXPECT_EQ(copyResult.GetErrorCode(), TTV_EC_UNIMPLEMENTED);

  copyResult = errorResult;

  EXPECT_FALSE(copyResult.IsSuccess());
  EXPECT_TRUE(copyResult.IsError());
  EXPECT_EQ(copyResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);
}

TEST(ResultTests, TestConvertErrorResult) {
  auto errorResult = ttv::MakeErrorResult(TTV_EC_UNKNOWN_ERROR);
  static_assert(std::is_same<decltype(errorResult), ttv::Result<void>>::value, "Unexpected result type.");

  EXPECT_FALSE(errorResult.IsSuccess());
  EXPECT_TRUE(errorResult.IsError());
  EXPECT_EQ(errorResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);

  ttv::Result<std::string> stringResult{errorResult};

  EXPECT_FALSE(stringResult.IsSuccess());
  EXPECT_TRUE(stringResult.IsError());
  EXPECT_EQ(stringResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);
}

TEST(ResultTests, TestAssignConvertErrorResult) {
  auto errorResult = ttv::MakeErrorResult(TTV_EC_UNKNOWN_ERROR);
  static_assert(std::is_same<decltype(errorResult), ttv::Result<void>>::value, "Unexpected result type.");

  EXPECT_FALSE(errorResult.IsSuccess());
  EXPECT_TRUE(errorResult.IsError());
  EXPECT_EQ(errorResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);

  ttv::Result<std::string> stringResult = ttv::MakeSuccessResult(std::string{"hello"});
  EXPECT_TRUE(stringResult.IsSuccess());
  EXPECT_FALSE(stringResult.IsError());
  EXPECT_EQ(stringResult.GetErrorCode(), TTV_EC_SUCCESS);
  EXPECT_EQ(stringResult.GetResult(), "hello");

  stringResult = errorResult;

  EXPECT_FALSE(stringResult.IsSuccess());
  EXPECT_TRUE(stringResult.IsError());
  EXPECT_EQ(stringResult.GetErrorCode(), TTV_EC_UNKNOWN_ERROR);
}

TEST(ResultTests, TestCopySuccessResult) {
  auto result = ttv::MakeSuccessResult(std::string{"hello"});
  static_assert(std::is_same<decltype(result), ttv::Result<std::string>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult(), "hello");
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  result.GetResult().append(" world");

  EXPECT_EQ(result.GetResult(), "hello world");

  ttv::Result<std::string> copiedResult{result};

  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "hello world");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult.GetResult().append("!");

  EXPECT_EQ(result.GetResult(), "hello world");
  EXPECT_EQ(copiedResult.GetResult(), "hello world!");
}

TEST(ResultTests, TestCopyAssignSuccessResult) {
  auto result = ttv::MakeSuccessResult(std::string{"hello"});
  static_assert(std::is_same<decltype(result), ttv::Result<std::string>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult(), "hello");
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  result.GetResult().append(" world");

  EXPECT_EQ(result.GetResult(), "hello world");

  ttv::Result<std::string> copiedResult = ttv::MakeSuccessResult("test");
  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "test");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult = result;

  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "hello world");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult.GetResult().append("!");

  EXPECT_EQ(result.GetResult(), "hello world");
  EXPECT_EQ(copiedResult.GetResult(), "hello world!");
}

TEST(ResultTests, TestMoveSuccessResult) {
  auto result = ttv::MakeSuccessResult(std::string{"hello"});
  static_assert(std::is_same<decltype(result), ttv::Result<std::string>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult(), "hello");
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  result.GetResult().append(" world");

  EXPECT_EQ(result.GetResult(), "hello world");

  ttv::Result<std::string> movedToResult = ttv::MakeSuccessResult("test");
  EXPECT_TRUE(movedToResult.IsSuccess());
  EXPECT_FALSE(movedToResult.IsError());
  EXPECT_EQ(movedToResult.GetResult(), "test");
  EXPECT_EQ(movedToResult.GetErrorCode(), TTV_EC_SUCCESS);

  movedToResult = std::move(result);

  EXPECT_TRUE(movedToResult.IsSuccess());
  EXPECT_FALSE(movedToResult.IsError());
  EXPECT_EQ(movedToResult.GetResult(), "hello world");
  EXPECT_EQ(movedToResult.GetErrorCode(), TTV_EC_SUCCESS);

  movedToResult.GetResult().append("!");

  EXPECT_EQ(result.GetResult(), "");
  EXPECT_EQ(movedToResult.GetResult(), "hello world!");
}

TEST(ResultTests, TestMoveAssignSuccessResult) {
  auto result = ttv::MakeSuccessResult(std::string{"hello"});
  static_assert(std::is_same<decltype(result), ttv::Result<std::string>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult(), "hello");
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  result.GetResult().append(" world");

  EXPECT_EQ(result.GetResult(), "hello world");

  ttv::Result<std::string> movedToResult{std::move(result)};

  EXPECT_TRUE(movedToResult.IsSuccess());
  EXPECT_FALSE(movedToResult.IsError());
  EXPECT_EQ(movedToResult.GetResult(), "hello world");
  EXPECT_EQ(movedToResult.GetErrorCode(), TTV_EC_SUCCESS);

  movedToResult.GetResult().append("!");

  EXPECT_EQ(result.GetResult(), "");
  EXPECT_EQ(movedToResult.GetResult(), "hello world!");
}

TEST(ResultTests, TestCopyConvertSuccessResult) {
  auto result = ttv::MakeSuccessResult("hello");
  static_assert(std::is_same<decltype(result), ttv::Result<const char*>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_TRUE(strcmp(result.GetResult(), "hello") == 0);
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  ttv::Result<std::string> copiedResult{result};

  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "hello");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult.GetResult().append(" world");

  EXPECT_TRUE(strcmp(result.GetResult(), "hello") == 0);
  EXPECT_EQ(copiedResult.GetResult(), "hello world");
}

TEST(ResultTests, TestCopyAssignConvertSuccessResult) {
  auto result = ttv::MakeSuccessResult("hello");
  static_assert(std::is_same<decltype(result), ttv::Result<const char*>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_TRUE(strcmp(result.GetResult(), "hello") == 0);
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  ttv::Result<std::string> copiedResult = ttv::MakeSuccessResult(std::string{"test"});
  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "test");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult = result;

  EXPECT_TRUE(copiedResult.IsSuccess());
  EXPECT_FALSE(copiedResult.IsError());
  EXPECT_EQ(copiedResult.GetResult(), "hello");
  EXPECT_EQ(copiedResult.GetErrorCode(), TTV_EC_SUCCESS);

  copiedResult.GetResult().append(" world");

  EXPECT_TRUE(strcmp(result.GetResult(), "hello") == 0);
  EXPECT_EQ(copiedResult.GetResult(), "hello world");
}

TEST(ResultTests, TestMoveConvertSuccessResult) {
  class Base {};

  class Derived : public Base {};

  auto derivedPointer = std::make_shared<Derived>();
  auto derivedPointerCopy = derivedPointer;
  auto result = ttv::MakeSuccessResult(std::move(derivedPointerCopy));
  static_assert(
    std::is_same<decltype(result), ttv::Result<std::shared_ptr<Derived>>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult().get(), derivedPointer.get());
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  ttv::Result<std::shared_ptr<Base>> movedResult{std::move(result)};

  EXPECT_TRUE(movedResult.IsSuccess());
  EXPECT_FALSE(movedResult.IsError());
  EXPECT_EQ(movedResult.GetResult().get(), derivedPointer.get());
  EXPECT_EQ(movedResult.GetErrorCode(), TTV_EC_SUCCESS);

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult().get(), nullptr);
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);
}

TEST(ResultTests, TestMoveAssignConvertSuccessResult) {
  class Base {};

  class Derived : public Base {};

  auto derivedPointer = std::make_shared<Derived>();
  auto derivedPointerCopy = derivedPointer;
  auto result = ttv::MakeSuccessResult(std::move(derivedPointerCopy));
  static_assert(
    std::is_same<decltype(result), ttv::Result<std::shared_ptr<Derived>>>::value, "Unexpected result type.");

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult().get(), derivedPointer.get());
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);

  ttv::Result<std::shared_ptr<Base>> movedResult = ttv::MakeSuccessResult(static_cast<std::shared_ptr<Base>>(nullptr));
  EXPECT_TRUE(movedResult.IsSuccess());
  EXPECT_FALSE(movedResult.IsError());
  EXPECT_EQ(movedResult.GetResult().get(), nullptr);
  EXPECT_EQ(movedResult.GetErrorCode(), TTV_EC_SUCCESS);

  movedResult = std::move(result);

  EXPECT_TRUE(movedResult.IsSuccess());
  EXPECT_FALSE(movedResult.IsError());
  EXPECT_EQ(movedResult.GetResult().get(), derivedPointer.get());
  EXPECT_EQ(movedResult.GetErrorCode(), TTV_EC_SUCCESS);

  EXPECT_TRUE(result.IsSuccess());
  EXPECT_FALSE(result.IsError());
  EXPECT_EQ(result.GetResult().get(), nullptr);
  EXPECT_EQ(result.GetErrorCode(), TTV_EC_SUCCESS);
}
