/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testeventschedulerfactory.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ibitslistener.h"
#include "twitchsdk/chat/internal/chatcommentmanager.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

#include <condition_variable>

#include <fstream>
#include <iostream>
#include <mutex>
#include <streambuf>

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestChatCommentListener : public ttv::chat::IChatCommentListener {
 public:
  TestChatCommentListener() {}

  virtual void ChatCommentManagerStateChanged(
    UserId userId, const std::string& vodId, IChatCommentManager::PlayingState state) override {}

  virtual void ChatCommentsReceived(
    UserId userId, const std::string& vodId, std::vector<ChatComment>&& messageList) override {}

  virtual void ChatCommentsErrorReceived(const std::string& errorMsg, TTV_ErrorCode ec) override {}
};

class TestBitsListener : public ttv::chat::IBitsListener {
 public:
  virtual void UserReceivedBits(const BitsReceivedEvent& bitsReceievedEvent) override {}

  virtual void UserSentBits(const BitsSentEvent& /*bitsSentEvent*/) override {}

  virtual void UserGainedBits(const uint32_t /*bitsBalance*/) override {}
};

class ShutdownTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    mTestEventScheduler = std::static_pointer_cast<TestEventScheduler>(mCoreApi->GetMainEventScheduler());

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "test";
    mUserInfo.displayName = "Test";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  }

  virtual void TearDownComponents() override { SdkBaseTest::TearDownComponents(); }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<ChatAPI> mChatApi;
  std::shared_ptr<TestEventScheduler> mTestEventScheduler;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(ShutdownTest, ChatCommentManagerShutdown) {
  std::mutex mutex;
  std::condition_variable cv;
  bool startedGetBitsConfigTask = false;
  bool finishedGetBitsConfigTask = false;
  bool ready = false;

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, "123");

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "channelId"}, "1001")
    .SetResponseHandler(
      [&mutex, &cv, &startedGetBitsConfigTask, &finishedGetBitsConfigTask, &ready, this](const std::string& /*url*/,
        ttv::HttpRequestType /*type*/, const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& body) {
        if (body.find("FetchChannelCheermotes") != std::string::npos) {
          {
            mTestEventScheduler->ScheduleTask({[&startedGetBitsConfigTask]() { startedGetBitsConfigTask = true; }});

            std::unique_lock<std::mutex> lock(mutex);
            cv.wait(lock, [&ready]() { return ready; });
          }

          // Return normal response for ChatGetBitsConfigTask
          std::ifstream response("chat/chatfetchchannelcheermotes.json", std::ifstream::in);
          std::string data((std::istreambuf_iterator<char>(response)), std::istreambuf_iterator<char>());

          mTestEventScheduler->ScheduleTask({[&finishedGetBitsConfigTask]() { finishedGetBitsConfigTask = true; }});

          return MockResponse::MockResponseContent(std::move(data), std::map<std::string, std::string>(), 200);
        }

        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 404);
      })
    .Done();

  std::shared_ptr<ChatCommentManager> chatCommentManager;

  // Result will hold onto a ref of chatCommentManager, so make sure it goes out of scope.
  {
    auto result = mChatApi->CreateChatCommentManager(12345, "123", std::make_shared<TestChatCommentListener>());
    ASSERT_FALSE(result.IsError());
    chatCommentManager = std::static_pointer_cast<ChatCommentManager>(result.GetResult());
  }

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500,
    [&startedGetBitsConfigTask]() { return startedGetBitsConfigTask; }, GetDefaultUpdateFunc()));

  chatCommentManager->Dispose();

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500,
    [&chatCommentManager]() { return chatCommentManager->GetState() == IComponent::State::Uninitialized; },
    GetDefaultUpdateFunc()));

  chatCommentManager.reset();

  {
    {
      std::unique_lock<std::mutex> lock(mutex);
      ready = true;
    }
    cv.notify_one();
  }

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500,
    [&finishedGetBitsConfigTask]() { return finishedGetBitsConfigTask; }, GetDefaultUpdateFunc()));
}

TEST_F(ShutdownTest, BitsStatusShutdown) {
  std::mutex mutex;
  std::condition_variable cv;
  bool startedGetBitsConfigTask = false;
  bool finishedGetBitsConfigTask = false;
  bool ready = false;

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseHandler(
      [&mutex, &cv, &startedGetBitsConfigTask, &finishedGetBitsConfigTask, &ready, this](const std::string& /*url*/,
        ttv::HttpRequestType /*type*/, const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& body) {
        if (body.find("FetchChannelCheermotes") != std::string::npos) {
          {
            mTestEventScheduler->ScheduleTask({[&startedGetBitsConfigTask]() { startedGetBitsConfigTask = true; }});

            std::unique_lock<std::mutex> lock(mutex);
            cv.wait(lock, [&ready]() { return ready; });
          }

          // Return normal response for ChatGetBitsConfigTask
          std::ifstream response("chat/chatfetchchannelcheermotes.json", std::ifstream::in);
          std::string data((std::istreambuf_iterator<char>(response)), std::istreambuf_iterator<char>());

          mTestEventScheduler->ScheduleTask({[&finishedGetBitsConfigTask]() { finishedGetBitsConfigTask = true; }});

          return MockResponse::MockResponseContent(std::move(data), std::map<std::string, std::string>(), 200);
        }

        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 404);
      })
    .Done();

  std::shared_ptr<BitsStatus> bitsStatus;

  // Result will hold onto a ref of bitsStatus, so make sure it goes out of scope.
  {
    std::shared_ptr<IBitsStatus> iBitsStatus;
    TTV_ErrorCode ec = mChatApi->CreateBitsStatus(12345, std::make_shared<TestBitsListener>(), iBitsStatus);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    bitsStatus = std::static_pointer_cast<BitsStatus>(iBitsStatus);
  }

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500000,
    [&startedGetBitsConfigTask]() { return startedGetBitsConfigTask; }, GetDefaultUpdateFunc()));

  bitsStatus->Dispose();

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500000,
    [&bitsStatus]() { return bitsStatus->GetState() == IComponent::State::Uninitialized; }, GetDefaultUpdateFunc()));

  bitsStatus.reset();

  {
    {
      std::unique_lock<std::mutex> lock(mutex);
      ready = true;
    }
    cv.notify_one();
  }

  ASSERT_TRUE(WaitUntilResultWithPollTask(mTestEventScheduler, 500000,
    [&finishedGetBitsConfigTask]() { return finishedGetBitsConfigTask; }, GetDefaultUpdateFunc()));
}
