/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "testwebsocket.h"

ttv::test::TestWebSocket::TestWebSocket() : mConnected(false), mShouldFail(false) {}

TTV_ErrorCode ttv::test::TestWebSocket::Connect() {
  if (mShouldFail) {
    return TTV_EC_SOCKET_CONNECT_FAILED;
  }

  if (mConnected) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mConnected = true;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestWebSocket::Disconnect() {
  if (!mConnected) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mConnected = false;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestWebSocket::Send(MessageType type, const uint8_t* buffer, size_t length) {
  TTV_ASSERT(type == MessageType::Text);
  TTV_ASSERT(buffer != nullptr);
  if (mShouldFail) {
    (void)Disconnect();
    return TTV_EC_SOCKET_SEND_ERROR;
  }

  if (mSentCallback != nullptr) {
    mSentCallback(std::string(reinterpret_cast<const char*>(buffer), length));
  }
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestWebSocket::Recv(MessageType& type, uint8_t* buffer, size_t length, size_t& received) {
  received = 0;
  type = MessageType::None;

  if (mShouldFail) {
    Disconnect();
    return TTV_EC_SOCKET_RECV_ERROR;
  }

  if (!mConnected) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  if (mReceivedPayloadQueue.empty()) {
    return TTV_EC_SOCKET_EWOULDBLOCK;
  }

  const auto& nextReceivedPayloadPair = mReceivedPayloadQueue.front();
  const auto& nextReceivedPayload = nextReceivedPayloadPair.first;
  auto payloadLength = nextReceivedPayload.length();

  if (payloadLength > length) {
    return TTV_EC_INVALID_BUFFER;
  } else {
    nextReceivedPayload.copy(reinterpret_cast<char*>(buffer), payloadLength, 0);
    received = payloadLength;
    type = MessageType::Text;

    auto callback = nextReceivedPayloadPair.second;
    if (callback != nullptr) {
      callback();
    }

    mReceivedPayloadQueue.pop();
  }
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestWebSocket::Peek(MessageType& type, size_t& length) {
  if (mShouldFail) {
    length = 0;
    type = MessageType::None;
    Disconnect();
    return TTV_EC_SOCKET_RECV_ERROR;
  }

  if (!mConnected) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  if (mReceivedPayloadQueue.empty()) {
    type = MessageType::None;
    length = 0;
    return TTV_EC_SOCKET_EWOULDBLOCK;
  } else {
    type = MessageType::Text;
    length = mReceivedPayloadQueue.front().first.length();
  }
  return TTV_EC_SUCCESS;
}

bool ttv::test::TestWebSocket::Connected() {
  return mConnected;
}
