/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/threadedeventscheduler.h"

using namespace ttv;
using namespace ttv::test;

namespace {
class ThreadedEventSchedulerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();
    mEventScheduler = std::make_shared<ThreadedEventScheduler>("ThreadedEventSchedulerTest");
  }

  virtual void TearDownComponents() override {
    mEventScheduler.reset();
    SdkBaseTest::TearDownComponents();
  }

  std::shared_ptr<ThreadedEventScheduler> mEventScheduler;
  int32_t mTasksRun = 0;
  bool mShutdown = false;
};
}  // namespace

TEST_F(ThreadedEventSchedulerTest, SimpleInsertsNoDelay) {
  for (int i = 0; i < 100; i++) {
    mEventScheduler->ScheduleTask({[this] { mTasksRun++; }});
  }

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mTasksRun == 100; }));
  mTasksRun = 0;

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
}

TEST_F(ThreadedEventSchedulerTest, SimpleInsertsWithDelay) {
  for (uint64_t i = 0; i < 100; i++) {
    mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, i});
  }

  ASSERT_TRUE(WaitUntilResult(200, [this]() { return mTasksRun == 100; }));
  mTasksRun = 0;

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
}

TEST_F(ThreadedEventSchedulerTest, StressCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 1000});
        ASSERT_FALSE(result.IsError());
        ASSERT_EQ(mEventScheduler->CancelTask(result.GetResult()), TTV_EC_SUCCESS);
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  ASSERT_FALSE(WaitUntilResult(200, [this]() { return mTasksRun > 0; }));
  ASSERT_EQ(mTasksRun, 0);
  mTasksRun = 0;

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
}

TEST_F(ThreadedEventSchedulerTest, StressHalfCancel) {
  // Stress test with 2000 tasks across 20 threads being scheduled on the EventScheduler with all of them being
  // cancelled
  std::vector<std::shared_ptr<IThread>> threads;

  int numThreads = 20;
  int tasksPerThread = 100;
  for (int i = 0; i < numThreads; i++) {
    auto insertFunc = [this, tasksPerThread]() {
      for (int i = 0; i < tasksPerThread; i++) {
        Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 100});
        ASSERT_FALSE(result.IsError());

        TaskId id = result.GetResult();
        if (id % 2 == 0) {
          ASSERT_EQ(mEventScheduler->CancelTask(id), TTV_EC_SUCCESS);
        }
      }
    };

    std::shared_ptr<IThread> thread;
    CreateThread(insertFunc, "thread " + std::to_string(i), thread);
    threads.push_back(thread);
  }

  for (auto thread : threads) {
    thread->Run();
  }

  ASSERT_EQ(threads.size(), numThreads);

  auto it = threads.begin();
  while (it != threads.end()) {
    (*it)->Join();
    it->reset();
    it = threads.erase(it);
  }
  ASSERT_EQ(threads.size(), 0);

  ASSERT_TRUE(WaitUntilResult(
    1000, [this, &numThreads, &tasksPerThread]() { return mTasksRun == (numThreads * tasksPerThread / 2); }));
  ASSERT_EQ(mTasksRun, (numThreads * tasksPerThread / 2));
  mTasksRun = 0;

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
}

TEST_F(ThreadedEventSchedulerTest, SortedOrder) {
  // Test the sorted order of the output (with some cancellation)
  std::vector<TaskId> tasksRun;

  Result<TaskId> result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(1); }, 160});
  ASSERT_TRUE(result.GetResult() == 1);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(2); }, 160});
  ASSERT_TRUE(result.GetResult() == 2);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(3); }, 40});
  ASSERT_TRUE(result.GetResult() == 3);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(4); }, 0});
  ASSERT_TRUE(result.GetResult() == 4);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(5); }, 40});
  ASSERT_TRUE(result.GetResult() == 5);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(6); }, 80});
  ASSERT_TRUE(result.GetResult() == 6);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(7); }, 0});
  ASSERT_TRUE(result.GetResult() == 7);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(8); }, 0});
  ASSERT_TRUE(result.GetResult() == 8);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(9); }, 160});
  ASSERT_TRUE(result.GetResult() == 9);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(10); }, 120});
  ASSERT_TRUE(result.GetResult() == 10);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(11); }, 120});
  ASSERT_TRUE(result.GetResult() == 11);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(12); }, 120});
  ASSERT_TRUE(result.GetResult() == 12);

  result = mEventScheduler->ScheduleTask({[&tasksRun] { tasksRun.emplace_back(13); }, 40});
  ASSERT_TRUE(result.GetResult() == 13);

  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_SUCCESS);
  ASSERT_EQ(mEventScheduler->CancelTask(11), TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(200, [&tasksRun]() { return tasksRun.size() == 11; }));
  ASSERT_EQ(tasksRun.size(), 11);

  std::array<int, 11> arr{{4, 7, 8, 3, 5, 13, 6, 10, 12, 2, 9}};

  for (int i = 0; i < 11; i++) {
    ASSERT_EQ(arr[i], tasksRun[i]);
  }

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));
}

TEST_F(ThreadedEventSchedulerTest, FailureCases) {
  // Test various failure cases

  ASSERT_EQ(mEventScheduler->CancelTask(0), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mEventScheduler->CancelTask(100), TTV_EC_OPERATION_FAILED);

  Result<TaskId> result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 0});
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 1);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mTasksRun == 1; }));
  ASSERT_EQ(mTasksRun, 1);

  ASSERT_EQ(mEventScheduler->CancelTask(1), TTV_EC_OPERATION_FAILED);
  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 100000});
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 2);
  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_SUCCESS);
  ASSERT_EQ(mEventScheduler->CancelTask(2), TTV_EC_OPERATION_FAILED);
  ASSERT_EQ(mTasksRun, 1);

  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 100000});
  ASSERT_FALSE(result.IsError());
  ASSERT_TRUE(result.GetResult() == 3);

  TTV_ErrorCode ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_SUCCESS);

  ASSERT_TRUE(WaitUntilResult(100, [this]() { return mShutdown; }));

  ASSERT_EQ(mTasksRun, 1);

  result = mEventScheduler->ScheduleTask({[this] { mTasksRun++; }, 1000});
  ASSERT_TRUE(result.IsError());
  ASSERT_EQ(result.GetErrorCode(), TTV_EC_NOT_INITIALIZED);

  ASSERT_EQ(mEventScheduler->CancelTask(3), TTV_EC_NOT_INITIALIZED);

  ec = mEventScheduler->Shutdown([this] { mShutdown = true; });
  ASSERT_EQ(ec, TTV_EC_NOT_INITIALIZED);
}
