/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/repositorytest.h"
#include "testutilities.h"
#include "twitchsdk/core/user/userrepository.h"

using namespace ttv;

namespace {
static const uint32_t kTestTimeoutMilliseconds = 2000;

void CheckUserInfo(TTV_ErrorCode ec, const UserInfo& userInfo) {
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ("twitch", userInfo.userName);
  ASSERT_EQ(12826, userInfo.userId);
}
}  // namespace

TEST_F(RepositoryTest, UserIdFetchGet) {
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "12826")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/getuserbyid.json")
    .Done();

  bool fetchChecked = false;

  std::function<void(const ErrorDetails&, const UserInfo&)> callback = [&fetchChecked](const ErrorDetails& errorDetails,
                                                                         const UserInfo& userInfo) {
    CheckUserInfo(errorDetails.ec, userInfo);
    fetchChecked = true;
  };

  userRepository->FetchUserInfoById(12826, callback);

  WaitUntilResult(kTestTimeoutMilliseconds, [&userRepository, &fetchChecked]() {
    userRepository->Update();
    return fetchChecked;
  });

  ASSERT_TRUE(fetchChecked);

  UserInfo userInfo;
  TTV_ErrorCode ec = userRepository->GetUserInfoById(12826, userInfo);
  CheckUserInfo(ec, userInfo);

  ec = userRepository->GetUserInfoByName("twitch", userInfo);
  CheckUserInfo(ec, userInfo);
}

TEST_F(RepositoryTest, UserNameFetchGet) {
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "login"}, "twitch")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("core/getuserbyname.json")
    .Done();

  bool fetchChecked = false;

  std::function<void(const ErrorDetails&, const UserInfo&)> callback = [&fetchChecked](const ErrorDetails& errorDetails,
                                                                         const UserInfo& userInfo) {
    CheckUserInfo(errorDetails.ec, userInfo);
    fetchChecked = true;
  };

  userRepository->FetchUserInfoByName("twitch", callback);

  WaitUntilResult(kTestTimeoutMilliseconds, [&userRepository, &fetchChecked]() {
    userRepository->Update();
    return fetchChecked;
  });

  ASSERT_TRUE(fetchChecked);

  UserInfo userInfo;
  TTV_ErrorCode ec = userRepository->GetUserInfoById(12826, userInfo);
  CheckUserInfo(ec, userInfo);

  ec = userRepository->GetUserInfoByName("twitch", userInfo);
  CheckUserInfo(ec, userInfo);
}

TEST_F(RepositoryTest, NoUserFetchAndGet) {
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);

  bool fetchChecked = false;

  std::function<void(const ErrorDetails&, const UserInfo&)> callback = [&fetchChecked](const ErrorDetails& errorDetails,
                                                                         const UserInfo& /*userInfo*/) {
    ASSERT_FALSE(TTV_SUCCEEDED(errorDetails.ec));
    fetchChecked = true;
  };

  userRepository->FetchUserInfoById(12826, callback);

  WaitUntilResult(kTestTimeoutMilliseconds, [&userRepository, &fetchChecked]() {
    userRepository->Update();
    return fetchChecked;
  });

  ASSERT_TRUE(fetchChecked);

  UserInfo userInfo;
  TTV_ErrorCode ec = userRepository->GetUserInfoById(12826, userInfo);

  ASSERT_FALSE(TTV_SUCCEEDED(ec));
}

TEST_F(RepositoryTest, UserRegisterUnregister) {
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);
  std::shared_ptr<User> user = userRepository->RegisterUser(12826);

  ASSERT_EQ(user, userRepository->GetUser(12826));

  userRepository->UnRegisterUser(12826);

  ASSERT_EQ(nullptr, userRepository->GetUser(12826));
}

TEST_F(RepositoryTest, UserRegisterUnregisterMultiple) {
  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  AddComponent(userRepository);
  std::vector<std::shared_ptr<User>> users;
  std::shared_ptr<User> user = userRepository->RegisterUser(12826);
  userRepository->RegisterUser(12826);
  TTV_ErrorCode ec = userRepository->GetUsers(users);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(user, userRepository->GetUser(12826));
  ASSERT_EQ(1, users.size());

  std::shared_ptr<User> user2 = userRepository->RegisterUser(12827);
  users.clear();
  ec = userRepository->GetUsers(users);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(user2, userRepository->GetUser(12827));
  ASSERT_EQ(2, users.size());

  ec = userRepository->UnRegisterUser(12826);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(nullptr, userRepository->GetUser(12826));

  users.clear();
  ec = userRepository->GetUsers(users);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(1, users.size());

  ec = userRepository->UnRegisterUser(12826);

  ASSERT_TRUE(TTV_FAILED(ec));

  users.clear();
  ec = userRepository->GetUsers(users);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(1, users.size());

  userRepository->UnRegisterUser(12827);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(nullptr, userRepository->GetUser(12827));

  users.clear();
  ec = userRepository->GetUsers(users);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(0, users.size());
}
