/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "usertestutility.h"

#include "testutilities.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

ttv::UserInfo ttv::test::UserTestUtility::DefaultUserInfo() const {
  UserInfo defaultUserInfo;
  defaultUserInfo.userName = "twitch";
  defaultUserInfo.displayName = "Twitch";
  defaultUserInfo.userId = 12345;
  defaultUserInfo.bio =
    "Twitch is the world's leading video platform and community for gamers with more than 100 million visitors per month. Our mission is to connect gamers around the world by allowing them to broadcast, watch, and chat from everywhere they play.";
  defaultUserInfo.logoImageUrl =
    "https://static-cdn.jtvnw.net/jtv_user_pictures/twitch-profile_image-42277a83919bd975-300x300.png";

  return defaultUserInfo;
}

void ttv::test::UserTestUtility::SetUpStubs(std::shared_ptr<TestHttpRequest> testHttpRequest) {
  testHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HttpRequestType::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("GetCurrentUser"))
    .SetResponseHandler([this](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                          const std::vector<ttv::HttpParam>& headerParams, const std::string& /*body*/) {
      auto headersIter = std::find_if(headerParams.begin(), headerParams.end(),
        [](const ttv::HttpParam& parameter) { return parameter.paramName == "Authorization"; });

      if (headersIter == headerParams.end()) {
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 401);
      }

      // Strip the "OAuth" prefix from teh value.
      auto authToken = headersIter->paramValue.substr(sizeof("OAuth ") - 1);
      auto userInfoIter = mUserInfosByAuthToken.find(authToken);
      if (userInfoIter == mUserInfosByAuthToken.end()) {
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 401);
      }

      json::FastWriter writer;
      return MockResponse::MockResponseContent(writer.write(JsonForUserInfo(userInfoIter->second)));
    })
    .Done();
}

void ttv::test::UserTestUtility::RegisterUserInfo(const std::string& authToken, const UserInfo& userInfo) {
  mUserInfosByAuthToken[authToken] = userInfo;
}

ttv::json::Value ttv::test::UserTestUtility::JsonForUserInfoKraken(const UserInfo& userInfo) {
  ttv::json::Value value;
  if (userInfo.userName != "")
    value["name"] = userInfo.userName;
  if (userInfo.displayName != "")
    value["display_name"] = userInfo.displayName;
  if (userInfo.userId != 0)
    value["_id"] = userInfo.userId;
  if (userInfo.bio != "")
    value["bio"] = userInfo.bio;
  if (userInfo.logoImageUrl != "")
    value["logo"] = userInfo.logoImageUrl;
  value["created_at"] = UnixTimestampToRFC3339String(userInfo.createdTimestamp);

  return value;
}

ttv::json::Value ttv::test::UserTestUtility::JsonForUserInfo(const UserInfo& userInfo) {
  ttv::json::Value value;

  value["data"] = ttv::json::Value(ttv::json::objectValue);
  value["data"]["currentUser"] = ttv::json::Value(ttv::json::objectValue);
  auto& currentUser = value["data"]["currentUser"];

  if (userInfo.userName != "")
    currentUser["login"] = userInfo.userName;
  if (userInfo.displayName != "")
    currentUser["displayName"] = userInfo.displayName;
  if (userInfo.userId != 0)
    currentUser["id"] = std::to_string(userInfo.userId);
  if (userInfo.bio != "")
    currentUser["description"] = userInfo.bio;
  if (userInfo.logoImageUrl != "")
    currentUser["profileImageURL"] = userInfo.logoImageUrl;

  currentUser["created_at"] = UnixTimestampToRFC3339String(userInfo.createdTimestamp);

  return value;
}
