/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/variant.h"

#include "gtest/gtest.h"

namespace {
class LifecycleTracker {
 public:
  LifecycleTracker()
      : mConstructorInvokeCount(0),
        mCopyConstructorInvokeCount(0),
        mCopyAssignmentInvokeCount(0),
        mMoveConstructorInvokeCount(0),
        mMoveAssignmentInvokeCount(0),
        mDestructorInvokeCount(0) {}

  void ConstructorCalled() { mConstructorInvokeCount++; }

  void CopyConstructorCalled() { mCopyConstructorInvokeCount++; }

  void CopyAssignmentCalled() { mCopyAssignmentInvokeCount++; }

  void MoveConstructorCalled() { mMoveConstructorInvokeCount++; }

  void MoveAssignmentCalled() { mMoveAssignmentInvokeCount++; }

  void DestructorCalled() { mDestructorInvokeCount++; }

  size_t GetConstructorInvokeCount() { return mConstructorInvokeCount; }

  size_t GetCopyConstructorInvokeCount() { return mCopyConstructorInvokeCount; }

  size_t GetCopyAssignmentInvokeCount() { return mCopyAssignmentInvokeCount; }

  size_t GetMoveConstructorInvokeCount() { return mMoveConstructorInvokeCount; }

  size_t GetMoveAssignmentInvokeCount() { return mMoveAssignmentInvokeCount; }

  size_t GetDestructorInvokeCount() { return mDestructorInvokeCount; }

  void ResetInvokeCounts() {
    mConstructorInvokeCount = 0;
    mCopyConstructorInvokeCount = 0;
    mCopyAssignmentInvokeCount = 0;
    mMoveConstructorInvokeCount = 0;
    mMoveAssignmentInvokeCount = 0;
    mDestructorInvokeCount = 0;
  }

 private:
  size_t mConstructorInvokeCount;
  size_t mCopyConstructorInvokeCount;
  size_t mCopyAssignmentInvokeCount;
  size_t mMoveConstructorInvokeCount;
  size_t mMoveAssignmentInvokeCount;
  size_t mDestructorInvokeCount;
};

class TrackedObject {
 public:
  TrackedObject(LifecycleTracker& tracker) : mTracker(tracker) { tracker.ConstructorCalled(); }

  TrackedObject(const TrackedObject& src) : mTracker(src.mTracker) { mTracker.CopyConstructorCalled(); }

  TrackedObject& operator=(const TrackedObject& src) {
    mTracker = src.mTracker;
    mTracker.CopyAssignmentCalled();
    return *this;
  }

  TrackedObject(TrackedObject&& src) : mTracker(src.mTracker) { mTracker.MoveConstructorCalled(); }

  TrackedObject& operator=(TrackedObject&& src) {
    mTracker = src.mTracker;
    mTracker.MoveAssignmentCalled();
    return *this;
  }

  ~TrackedObject() { mTracker.DestructorCalled(); }

 private:
  LifecycleTracker& mTracker;
};

class DummyObject {};

class SecondDummyObject {};

template <typename... Subtypes>
void TestBasicLifeCycle() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> variant{TrackedObject{tracker}};

    ASSERT_TRUE(variant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestVariantCopyConstructor() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{tracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{sourceVariant};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 3);
}

template <typename... Subtypes>
void TestVariantCopyAssignment() {
  LifecycleTracker sourceTracker;
  LifecycleTracker destinationTracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{sourceTracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{destinationTracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 1);

    destinationVariant = sourceVariant;

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());
    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 1);

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 3);

  ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestVariantMoveConstructor() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{tracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{std::move(sourceVariant)};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 2);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 2);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 3);
}

template <typename... Subtypes>
void TestVariantMoveAssignment() {
  LifecycleTracker sourceTracker;
  LifecycleTracker destinationTracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{sourceTracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{destinationTracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 1);

    destinationVariant = std::move(sourceVariant);

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 2);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 1);

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 2);
  ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 3);

  ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestValueCopyAssignment() {
  LifecycleTracker sourceTracker;
  LifecycleTracker destinationTracker;
  {
    TrackedObject sourceObject{sourceTracker};
    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 0);

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{destinationTracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 1);

    destinationVariant = sourceObject;

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 0);

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 2);

  ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestValueMoveAssignment() {
  LifecycleTracker sourceTracker;
  LifecycleTracker destinationTracker;
  {
    TrackedObject sourceObject{sourceTracker};
    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 0);

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{destinationTracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 1);

    destinationVariant = std::move(sourceObject);

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 0);

    ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(sourceTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(sourceTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(sourceTracker.GetDestructorInvokeCount(), 2);

  ASSERT_EQ(destinationTracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(destinationTracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(destinationTracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestCopySourceOverwriteTypeWithVariant() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{tracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{DummyObject{}};

    ASSERT_TRUE(destinationVariant.template Is<DummyObject>());

    sourceVariant = destinationVariant;

    ASSERT_TRUE(sourceVariant.template Is<DummyObject>());
    ASSERT_TRUE(destinationVariant.template Is<DummyObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestMoveSourceOverwriteTypeWithVariant() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{TrackedObject{tracker}};

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    ttv::Variant<Subtypes...> destinationVariant{DummyObject{}};

    ASSERT_TRUE(destinationVariant.template Is<DummyObject>());

    sourceVariant = std::move(destinationVariant);

    ASSERT_TRUE(sourceVariant.template Is<DummyObject>());
    ASSERT_TRUE(destinationVariant.template Is<DummyObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 2);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 2);
}

template <typename... Subtypes>
void TestCopyDestinationOverwriteTypeWithVariant() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{DummyObject{}};

    ASSERT_TRUE(sourceVariant.template Is<DummyObject>());

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{tracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    sourceVariant = destinationVariant;

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());
    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 3);
}

template <typename... Subtypes>
void TestMoveDestinationOverwriteTypeWithVariant() {
  LifecycleTracker tracker;
  {
    ttv::Variant<Subtypes...> sourceVariant{DummyObject{}};

    ASSERT_TRUE(sourceVariant.template Is<DummyObject>());

    ttv::Variant<Subtypes...> destinationVariant{TrackedObject{tracker}};

    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);

    sourceVariant = std::move(destinationVariant);

    ASSERT_TRUE(sourceVariant.template Is<TrackedObject>());
    ASSERT_TRUE(destinationVariant.template Is<TrackedObject>());

    ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
    ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
    ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 2);
    ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
    ASSERT_EQ(tracker.GetDestructorInvokeCount(), 1);
  }

  ASSERT_EQ(tracker.GetConstructorInvokeCount(), 1);
  ASSERT_EQ(tracker.GetCopyConstructorInvokeCount(), 0);
  ASSERT_EQ(tracker.GetCopyAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetMoveConstructorInvokeCount(), 2);
  ASSERT_EQ(tracker.GetMoveAssignmentInvokeCount(), 0);
  ASSERT_EQ(tracker.GetDestructorInvokeCount(), 3);
}
}  // namespace

TEST(VariantTests, TestBasicLifecycle) {
  TestBasicLifeCycle<TrackedObject>();

  TestBasicLifeCycle<TrackedObject, DummyObject>();
  TestBasicLifeCycle<DummyObject, TrackedObject>();

  TestBasicLifeCycle<TrackedObject, DummyObject, SecondDummyObject>();
  TestBasicLifeCycle<DummyObject, TrackedObject, SecondDummyObject>();
  TestBasicLifeCycle<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestVariantCopyConstructor) {
  TestVariantCopyConstructor<TrackedObject>();

  TestVariantCopyConstructor<TrackedObject, DummyObject>();
  TestVariantCopyConstructor<DummyObject, TrackedObject>();

  TestVariantCopyConstructor<TrackedObject, DummyObject, SecondDummyObject>();
  TestVariantCopyConstructor<DummyObject, TrackedObject, SecondDummyObject>();
  TestVariantCopyConstructor<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestVariantCopyAssignment) {
  TestVariantCopyAssignment<TrackedObject>();

  TestVariantCopyAssignment<TrackedObject, DummyObject>();
  TestVariantCopyAssignment<DummyObject, TrackedObject>();

  TestVariantCopyAssignment<TrackedObject, DummyObject, SecondDummyObject>();
  TestVariantCopyAssignment<DummyObject, TrackedObject, SecondDummyObject>();
  TestVariantCopyAssignment<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestVariantMoveConstructor) {
  TestVariantMoveConstructor<TrackedObject>();

  TestVariantMoveConstructor<TrackedObject, DummyObject>();
  TestVariantMoveConstructor<DummyObject, TrackedObject>();

  TestVariantMoveConstructor<TrackedObject, DummyObject, SecondDummyObject>();
  TestVariantMoveConstructor<DummyObject, TrackedObject, SecondDummyObject>();
  TestVariantMoveConstructor<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestVariantMoveAssignment) {
  TestVariantMoveAssignment<TrackedObject>();

  TestVariantMoveAssignment<TrackedObject, DummyObject>();
  TestVariantMoveAssignment<DummyObject, TrackedObject>();

  TestVariantMoveAssignment<TrackedObject, DummyObject, SecondDummyObject>();
  TestVariantMoveAssignment<DummyObject, TrackedObject, SecondDummyObject>();
  TestVariantMoveAssignment<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestValueCopyAssignment) {
  TestValueCopyAssignment<TrackedObject>();

  TestValueCopyAssignment<TrackedObject, DummyObject>();
  TestValueCopyAssignment<DummyObject, TrackedObject>();

  TestValueCopyAssignment<TrackedObject, DummyObject, SecondDummyObject>();
  TestValueCopyAssignment<DummyObject, TrackedObject, SecondDummyObject>();
  TestValueCopyAssignment<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestValueMoveAssignment) {
  TestValueMoveAssignment<TrackedObject>();

  TestValueMoveAssignment<TrackedObject, DummyObject>();
  TestValueMoveAssignment<DummyObject, TrackedObject>();

  TestValueMoveAssignment<TrackedObject, DummyObject, SecondDummyObject>();
  TestValueMoveAssignment<DummyObject, TrackedObject, SecondDummyObject>();
  TestValueMoveAssignment<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestCopySourceOverwriteTypeWithVariant) {
  TestCopySourceOverwriteTypeWithVariant<TrackedObject, DummyObject>();
  TestCopySourceOverwriteTypeWithVariant<DummyObject, TrackedObject>();

  TestCopySourceOverwriteTypeWithVariant<TrackedObject, DummyObject, SecondDummyObject>();
  TestCopySourceOverwriteTypeWithVariant<DummyObject, TrackedObject, SecondDummyObject>();
  TestCopySourceOverwriteTypeWithVariant<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestMoveSourceOverwriteTypeWithVariant) {
  TestMoveSourceOverwriteTypeWithVariant<TrackedObject, DummyObject>();
  TestMoveSourceOverwriteTypeWithVariant<DummyObject, TrackedObject>();

  TestMoveSourceOverwriteTypeWithVariant<TrackedObject, DummyObject, SecondDummyObject>();
  TestMoveSourceOverwriteTypeWithVariant<DummyObject, TrackedObject, SecondDummyObject>();
  TestMoveSourceOverwriteTypeWithVariant<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestCopyDestinationOverwriteTypeWithVariant) {
  TestCopyDestinationOverwriteTypeWithVariant<TrackedObject, DummyObject>();
  TestCopyDestinationOverwriteTypeWithVariant<DummyObject, TrackedObject>();

  TestCopyDestinationOverwriteTypeWithVariant<TrackedObject, DummyObject, SecondDummyObject>();
  TestCopyDestinationOverwriteTypeWithVariant<DummyObject, TrackedObject, SecondDummyObject>();
  TestCopyDestinationOverwriteTypeWithVariant<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestMoveDestinationOverwriteTypeWithVariant) {
  TestMoveDestinationOverwriteTypeWithVariant<TrackedObject, DummyObject>();
  TestMoveDestinationOverwriteTypeWithVariant<DummyObject, TrackedObject>();

  TestMoveDestinationOverwriteTypeWithVariant<TrackedObject, DummyObject, SecondDummyObject>();
  TestMoveDestinationOverwriteTypeWithVariant<DummyObject, TrackedObject, SecondDummyObject>();
  TestMoveDestinationOverwriteTypeWithVariant<DummyObject, SecondDummyObject, TrackedObject>();
}

TEST(VariantTests, TestConstVariant) {
  const ttv::Variant<DummyObject, SecondDummyObject> variant{DummyObject{}};

  ASSERT_TRUE(variant.Is<DummyObject>());
  ASSERT_FALSE(variant.Is<SecondDummyObject>());

  const DummyObject& objectRef = variant.As<DummyObject>();

  // Just making sure this doesn't assert or throw or anything.
  (void)objectRef;
}

TEST(VariantTests, TestNestedVariant) {
  ttv::Variant<DummyObject, SecondDummyObject> childVariant{DummyObject{}};

  ASSERT_TRUE(childVariant.Is<DummyObject>());
  ASSERT_FALSE(childVariant.Is<SecondDummyObject>());

  ttv::Variant<DummyObject, ttv::Variant<DummyObject, SecondDummyObject>> parentVariant{DummyObject{}};

  ASSERT_TRUE(parentVariant.Is<DummyObject>());
  ASSERT_FALSE((parentVariant.Is<ttv::Variant<DummyObject, SecondDummyObject>>()));

  parentVariant = childVariant;

  ASSERT_FALSE(parentVariant.Is<DummyObject>());
  ASSERT_TRUE((parentVariant.Is<ttv::Variant<DummyObject, SecondDummyObject>>()));
}
