/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/cfsocket.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/thread.h"

#include <sys/errno.h>
#include <sys/socket.h>

ttv::CFSocket::CFSocket(const std::string& host, uint32_t port, bool useSsl)
    : mHostName(host),
      mPort(port),
      mTotalSent(0),
      mTotalReceived(0),
      mUseSsl(useSsl),
      mReadStream(nullptr),
      mWriteStream(nullptr) {}

ttv::CFSocket::~CFSocket() {
  (void)Disconnect();
}

TTV_ErrorCode ttv::CFSocket::Connect() {
  TTV_ASSERT(!Connected());
  if (Connected()) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mTotalSent = 0;
  mTotalReceived = 0;

  CFStringRef hostString = CFStringCreateWithCString(kCFAllocatorDefault, mHostName.c_str(), kCFStringEncodingUTF8);
  if (hostString == nullptr) {
    return TTV_EC_SOCKET_HOST_NOT_FOUND;
  }
  CFStreamCreatePairWithSocketToHost(kCFAllocatorDefault, hostString, mPort, &mReadStream, &mWriteStream);
  CFRelease(hostString);

  if (mReadStream == nullptr || mWriteStream == nullptr) {
    ttv::trace::Message("Socket", MessageLevel::Error, "Failed to create CFStream pair.");
    (void)Disconnect();
    return TTV_EC_SOCKET_CREATE_FAILED;
  }

  // Set TLS/SSL properties
  CFReadStreamSetProperty(mReadStream, kCFStreamPropertySocketSecurityLevel,
    mUseSsl ? kCFStreamSocketSecurityLevelNegotiatedSSL : kCFStreamSocketSecurityLevelNone);
  CFWriteStreamSetProperty(mWriteStream, kCFStreamPropertySocketSecurityLevel,
    mUseSsl ? kCFStreamSocketSecurityLevelNegotiatedSSL : kCFStreamSocketSecurityLevelNone);

  // Open streams
  Boolean didOpenRead = CFReadStreamOpen(mReadStream);
  Boolean didOpenWrite = CFWriteStreamOpen(mWriteStream);

  if (!(didOpenRead && didOpenWrite)) {
    ttv::trace::Message("Socket", MessageLevel::Error, "Failed to open CFStream.");
    if (didOpenRead || didOpenWrite)
      (void)Disconnect();
    return TTV_EC_SOCKET_CONNECT_FAILED;
  }

  while (CFWriteStreamGetStatus(mWriteStream) == kCFStreamStatusOpening ||
         CFReadStreamGetStatus(mReadStream) == kCFStreamStatusOpening) {
    ttv::Sleep(10);
  }

  if (!(CFWriteStreamGetStatus(mWriteStream) == kCFStreamStatusOpen &&
        CFReadStreamGetStatus(mReadStream) == kCFStreamStatusOpen)) {
    ttv::trace::Message("Socket", MessageLevel::Error, "Failed to open CFStream.");
    (void)Disconnect();
    return TTV_EC_SOCKET_CONNECT_FAILED;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CFSocket::Disconnect() {
  if (mReadStream != nullptr) {
    CFReadStreamClose(mReadStream);
    CFRelease(mReadStream);
    mReadStream = nullptr;
  }

  if (mWriteStream != nullptr) {
    CFWriteStreamClose(mWriteStream);
    CFRelease(mWriteStream);
    mWriteStream = nullptr;
  }

  return TTV_EC_SUCCESS;
}

bool ttv::CFSocket::Connected() {
  return mReadStream != nullptr && mWriteStream != nullptr;
}

uint64_t ttv::CFSocket::TotalSent() {
  return mTotalSent;
}

uint64_t ttv::CFSocket::TotalReceived() {
  return mTotalReceived;
}

TTV_ErrorCode ttv::CFSocket::Send(const uint8_t* buffer, size_t length, size_t& sent) {
  assert(buffer);
  assert(length > 0);

  if (mWriteStream == nullptr) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  sent = 0;
  CFIndex bytesWritten = CFWriteStreamWrite(mWriteStream, buffer, static_cast<CFIndex>(length));
  if (bytesWritten < 0) {
    // An error has occurred
    CFStreamError error = CFWriteStreamGetError(mWriteStream);
    ttv::trace::Message("Socket", MessageLevel::Error, "Error sending to CFWriteStream. Domain = %d Error = %d",
      error.domain, error.error);
    ec = TTV_EC_SOCKET_SEND_ERROR;

    switch (CFWriteStreamGetStatus(mWriteStream)) {
      case kCFStreamStatusNotOpen:
      case kCFStreamStatusOpening:
        ec = TTV_EC_SOCKET_ENOTCONN;
        break;
      case kCFStreamStatusClosed:
        ec = TTV_EC_SOCKET_ECONNABORTED;
        break;
      case kCFStreamStatusWriting:
      case kCFStreamStatusReading:
      case kCFStreamStatusOpen:
      case kCFStreamStatusAtEnd:
      case kCFStreamStatusError:
        break;
    }
  } else if (bytesWritten == 0) {
    // The connection has dropped
    ec = TTV_EC_SOCKET_ECONNABORTED;
  } else {
    sent = static_cast<size_t>(bytesWritten);
    mTotalSent += static_cast<uint64_t>(bytesWritten);
  }

  if (TTV_FAILED(ec) && ec != TTV_EC_SOCKET_ENOTCONN && ec != TTV_EC_SOCKET_EWOULDBLOCK) {
    (void)Disconnect();
  }

  return ec;
}

TTV_ErrorCode ttv::CFSocket::Recv(uint8_t* buffer, size_t length, size_t& received) {
  assert(buffer);

  if (mReadStream == nullptr) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  received = 0;
  if (CFReadStreamHasBytesAvailable(mReadStream)) {
    CFIndex bytesRead = CFReadStreamRead(mReadStream, buffer, static_cast<CFIndex>(length));
    if (bytesRead < 0) {
      // An error has occurred
      CFStreamError error = CFReadStreamGetError(mReadStream);
      ttv::trace::Message("Socket", MessageLevel::Error, "Error receiving from CFReadStream. Domain = %d Error = %d",
        error.domain, error.error);
      ec = TTV_EC_SOCKET_RECV_ERROR;
    } else if (bytesRead == 0) {
      // The connection has dropped
      ec = TTV_EC_SOCKET_ECONNABORTED;
    } else {
      received = static_cast<size_t>(bytesRead);
      mTotalReceived += static_cast<uint64_t>(bytesRead);
    }
  } else {
    // Check socket status
    switch (CFReadStreamGetStatus(mReadStream)) {
      case kCFStreamStatusNotOpen:
      case kCFStreamStatusOpening:
        ec = TTV_EC_SOCKET_ENOTCONN;
        break;
      case kCFStreamStatusWriting:
      case kCFStreamStatusReading:
      case kCFStreamStatusOpen:
      case kCFStreamStatusAtEnd:
        ec = TTV_EC_SOCKET_EWOULDBLOCK;
        break;
      case kCFStreamStatusError:
        ec = TTV_EC_SOCKET_RECV_ERROR;
        break;
      case kCFStreamStatusClosed:
        ec = TTV_EC_SOCKET_ECONNABORTED;
        break;
    }
  }

  if (TTV_FAILED(ec) && ec != TTV_EC_SOCKET_ENOTCONN && ec != TTV_EC_SOCKET_EWOULDBLOCK) {
    (void)Disconnect();
  }

  return ec;
}

bool ttv::RawCFSocketFactory::IsProtocolSupported(const std::string& protocol) {
  return protocol == "" || protocol == "tcp";
}

TTV_ErrorCode ttv::RawCFSocketFactory::CreateSocket(const std::string& uri, std::shared_ptr<ttv::ISocket>& result) {
  result.reset();

  Uri url(uri);

  if (IsProtocolSupported(url.GetProtocol())) {
    uint32_t port = 0;
    if (!url.GetPort(port)) {
      return TTV_EC_INVALID_ARG;
    }

    result = std::make_shared<CFSocket>(url.GetHostName(), port, false);

    return TTV_EC_SUCCESS;
  }

  return TTV_EC_UNIMPLEMENTED;
}

bool ttv::SecureCFSocketFactory::IsProtocolSupported(const std::string& protocol) {
  return protocol == "tls" || protocol == "ssl";
}

TTV_ErrorCode ttv::SecureCFSocketFactory::CreateSocket(const std::string& uri, std::shared_ptr<ttv::ISocket>& result) {
  result.reset();

  Uri url(uri);

  if (IsProtocolSupported(url.GetProtocol())) {
    uint32_t port = 0;
    if (!url.GetPort(port)) {
      return TTV_EC_INVALID_ARG;
    }

    result = std::make_shared<CFSocket>(url.GetHostName(), port, true);

    return TTV_EC_SUCCESS;
  }

  return TTV_EC_UNIMPLEMENTED;
}
