package tv.twitch;

public class CoreAPI extends NativeProxy implements IModule {
    static { CoreErrorCode.forceClassInit(); }

    public interface LogInCallback { void invoke(ErrorCode ec, UserInfo userInfo); }
    public interface LogOutCallback { void invoke(ErrorCode ec); }
    public interface FetchUserInfoCallback { void invoke(ErrorCode ec, UserInfo userInfo); }
    public interface FetchChannelInfoCallback { void invoke(ErrorCode ec, ChannelInfo channelInfo); }
    public interface FetchStreamInfoCallback { void invoke(ErrorCode ec, StreamInfo streamInfo); }

    /**
     * Legacy Constructor
     * @Deprecated use {@link CoreAPI(IJniThreadChecker)}
     */
    @Deprecated
    public CoreAPI() {
        super(PassThroughJniThreadValidator.INSTANCE);
    }

    public CoreAPI(IJniThreadChecker jniThreadChecker) { super(new JniThreadValidator(jniThreadChecker)); }

    private native long CreateNativeInstance();
    // AND-9927 DisposeNativeInstance doesn't use a ScopedJavaEnvironmentCacher, so it's safe to call from any
    // thread
    private native void DisposeNativeInstance(long nativeObjectPointer);

    private native ModuleState GetState(long nativeObjectPointer);
    private native String GetModuleName(long nativeObjectPointer);
    private native ErrorCode Initialize(long nativeObjectPointer, InitializeCallback callback);
    private native ErrorCode Shutdown(long nativeObjectPointer, ShutdownCallback callback);
    private native ErrorCode Update(long nativeObjectPointer);

    private native ErrorCode SetListener(long nativeObjectPointer, ICoreAPIListener listener);
    private native ErrorCode GetRequiredOAuthScopes(
        long nativeObjectPointer, ResultContainer<String[]> modules, ResultContainer<String[]> scopes);
    private native ErrorCode LogIn(long nativeObjectPointer, String oauthToken, LogInCallback callback);
    private native ErrorCode LogOut(long nativeObjectPointer, int userId, LogOutCallback callback);
    private native ErrorCode ConnectPubSub(long nativeObjectPointer, int userId);
    private native ErrorCode DisconnectPubSub(long nativeObjectPointer, int userId);
    private native ErrorCode SetGlobalSetting(long nativeObjectPointer, String key, String value);
    private native ErrorCode RemoveGlobalSetting(long nativeObjectPointer, String key);
    private native ErrorCode GetGlobalSetting(long nativeObjectPointer, String key, ResultContainer<String> value);
    private native ErrorCode GetSubscribedPubsubTopics(long nativeObjectPointer, ResultContainer<String[]> topics);
    private native void CrashAbort(long nativeObjectPointer);
    private native ErrorCode SetLocalLanguage(long nativeObjectPointer, String language);
    private native ErrorCode GetLocalLanguage(long nativeObjectPointer, ResultContainer<String> result);
    private native ErrorCode FetchUserInfoById(long nativeObjectPointer, int userId, FetchUserInfoCallback callback);
    private native ErrorCode FetchUserInfoByName(
        long nativeObjectPointer, String userName, FetchUserInfoCallback callback);
    private native ErrorCode FetchChannelInfoById(
        long nativeObjectPointer, int channelId, FetchChannelInfoCallback callback);
    private native ErrorCode FetchChannelInfoByName(
        long nativeObjectPointer, String channelName, FetchChannelInfoCallback callback);
    private native ErrorCode FetchStreamInfoById(
        long nativeObjectPointer, int channelId, FetchStreamInfoCallback callback);
    private native ErrorCode FetchStreamInfoByName(
        long nativeObjectPointer, String channelName, FetchStreamInfoCallback callback);
    private native ErrorCode CreateChannelStatus(long nativeObjectPointer, IJniThreadValidator jniThreadValidator,
        int userId, int channelId, IChannelListener listener, ResultContainer<IChannelStatus> result);
    private native ErrorCode CreateDashboardActivityStatus(long nativeObjectPointer, int userId, int channelId,
        IDashboardActivityListener listener, ResultContainer<IDashboardActivityStatus> result);
    private native ErrorCode CreateGenericSubscriberStatus(long nativeObjectPointer, int userId, String topic,
        IGenericSubscriberListener listener, ResultContainer<IGenericSubscriberStatus> result);
    // AND-9927 It is safe to dispose of an IChannelStatus from any thread.
    private native ErrorCode DisposeChannelStatus(long nativeObjectPointer, IChannelStatus channelStatus);
    // AND-9927 It is safe to dispose of an IDashboardActivityStatus from any thread.
    private native ErrorCode DisposeDashboardActivityStatus(
        long nativeObjectPointer, IDashboardActivityStatus channelStatus);
    // AND-9927 It is safe to dispose of an IGenericSubscriberStatus from any thread.
    private native ErrorCode DisposeGenericSubscriberStatus(
        long nativeObjectPointer, IGenericSubscriberStatus channelStatus);

    @Override
    protected long createNativeInstance() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return CreateNativeInstance();
            }
        });
    }

    @Override
    protected void disposeNativeInstance(long nativeObjectPointer) {
        // AND-9927 DisposeNativeInstance doesn't use a ScopedJavaEnvironmentCacher, so it's safe to call from any
        // thread
        DisposeNativeInstance(nativeObjectPointer);
    }

    @Override
    public ModuleState getState() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ModuleState>() {
            @Override
            public ModuleState call() {
                return GetState(getNativeObjectPointer());
            }
        });
    }

    @Override
    public String getModuleName() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<String>() {
            @Override
            public String call() {
                return GetModuleName(getNativeObjectPointer());
            }
        });
    }

    @Override
    public ErrorCode initialize(final InitializeCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Initialize(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode shutdown(final ShutdownCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Shutdown(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode update() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Update(getNativeObjectPointer());
            }
        });
    }

    public ErrorCode setLocalLanguage(final String language) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetLocalLanguage(getNativeObjectPointer(), language);
            }
        });
    }

    public ErrorCode getLocalLanguage(final ResultContainer<String> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetLocalLanguage(getNativeObjectPointer(), result);
            }
        });
    }

    public ErrorCode fetchUserInfo(final int userId, final FetchUserInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchUserInfoById(getNativeObjectPointer(), userId, callback);
            }
        });
    }

    public ErrorCode fetchUserInfo(final String username, final FetchUserInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchUserInfoByName(getNativeObjectPointer(), username, callback);
            }
        });
    }

    public ErrorCode fetchChannelInfoById(final int channelId, final FetchChannelInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelInfoById(getNativeObjectPointer(), channelId, callback);
            }
        });
    }

    public ErrorCode fetchChannelInfoByName(final String channelName, final FetchChannelInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelInfoByName(getNativeObjectPointer(), channelName, callback);
            }
        });
    }

    public ErrorCode fetchStreamInfoById(final int channelId, final FetchStreamInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchStreamInfoById(getNativeObjectPointer(), channelId, callback);
            }
        });
    }

    public ErrorCode fetchStreamInfoByName(final String channelName, final FetchStreamInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchStreamInfoByName(getNativeObjectPointer(), channelName, callback);
            }
        });
    }

    public ErrorCode createChannelStatus(final int userId, final int channelId, final IChannelListener listener,
        final ResultContainer<IChannelStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChannelStatus(
                    getNativeObjectPointer(), mJniThreadValidator, userId, channelId, listener, result);
            }
        });
    }

    public ErrorCode createDashboardActivityStatus(final int userId, final int channelId,
        final IDashboardActivityListener listener, final ResultContainer<IDashboardActivityStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateDashboardActivityStatus(getNativeObjectPointer(), userId, channelId, listener, result);
            }
        });
    }

    public ErrorCode createGenericSubscriberStatus(final int userId, final String topic,
        final IGenericSubscriberListener listener, final ResultContainer<IGenericSubscriberStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateGenericSubscriberStatus(getNativeObjectPointer(), userId, topic, listener, result);
            }
        });
    }

    public ErrorCode disposeChannelStatus(IChannelStatus channelStatus) {
        // AND-9927 It is safe to dispose of an IChannelStatus from any thread.
        return DisposeChannelStatus(getNativeObjectPointer(), channelStatus);
    }

    public ErrorCode disposeDashboardActivityStatus(IDashboardActivityStatus status) {
        // AND-9927 It is safe to dispose of an IDashboardActivityStatus from any thread.
        return DisposeDashboardActivityStatus(getNativeObjectPointer(), status);
    }

    public ErrorCode disposeGenericSubscriberStatus(IGenericSubscriberStatus status) {
        // AND-9927 It is safe to dispose of an IGenericSubscriberStatus from any thread.
        return DisposeGenericSubscriberStatus(getNativeObjectPointer(), status);
    }

    public void setListener(final ICoreAPIListener listener) {
        mJniThreadValidator.callJniRunnable(new Runnable() {
            @Override
            public void run() {
                SetListener(getNativeObjectPointer(), listener);
            }
        });
    }

    public ErrorCode getRequiredOAuthScopes(
        final ResultContainer<String[]> modules, final ResultContainer<String[]> scopes) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetRequiredOAuthScopes(getNativeObjectPointer(), modules, scopes);
            }
        });
    }

    public ErrorCode logIn(final String oauthToken, final LogInCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return LogIn(getNativeObjectPointer(), oauthToken, callback);
            }
        });
    }

    public ErrorCode logOut(final int userId, final LogOutCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return LogOut(getNativeObjectPointer(), userId, callback);
            }
        });
    }

    public ErrorCode connectPubSub(final int userId) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return ConnectPubSub(getNativeObjectPointer(), userId);
            }
        });
    }

    public ErrorCode disconnectPubSub(final int userId) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return DisconnectPubSub(getNativeObjectPointer(), userId);
            }
        });
    }

    public ErrorCode setGlobalSetting(final String key, final String value) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetGlobalSetting(getNativeObjectPointer(), key, value);
            }
        });
    }

    public ErrorCode removeGlobalSetting(final String key) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RemoveGlobalSetting(getNativeObjectPointer(), key);
            }
        });
    }

    public ErrorCode getGlobalSetting(final String key, final ResultContainer<String> value) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetGlobalSetting(getNativeObjectPointer(), key, value);
            }
        });
    }

    public ErrorCode getSubscribedPubsubTopics(final ResultContainer<String[]> topics) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetSubscribedPubsubTopics(getNativeObjectPointer(), topics);
            }
        });
    }

    public void crashAbort() {
        mJniThreadValidator.callJniRunnable(new Runnable() {
            @Override
            public void run() {
                CrashAbort(getNativeObjectPointer());
            }
        });
    }
}
