package tv.twitch;

/**
 * Interface that describes a serial dispatch queue where tasks/events can be scheduled to be run immediately or with
 * some delay. Tasks (code blocks) added to the IEventScheduler will be run in ascending timestamp priority order, with
 * older tasks executing first. When tasks with the same timestamp are added into the queue, the task added earlier will
 * be executed first. Each task that is added into IEventScheduler can optionally get back a TaskId that uniquely
 * identifies the task in IEventScheduler. This TaskId can be used later to cancel the task (if it already hasn't been
 * executed).
 *
 * For IEventScheduler implementations, Shutdown() must be called before the IEventScheduler is destroyed.
 */
public interface IEventScheduler {
    /**
     * Adds a task into the IEventScheduler with the current time + some delay.
     * The task will be run before other pending tasks with timestamp `current time + delay` or later.
     * Should be called when the IEventScheduler is in the Running state.
     *
     * @param[in] taskParams Parameters (taskFunction, delay, name) of the task to be added into the IEventScheduler.
     * @return
     *   - The unique id of the task that can be used to cancel the task later with `CancelTask`.
     *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has not yet been initialized or has already been shutdown.
     *   - TTV_EC_INVALID_ARG: taskFunction is nullptr and the task was not added into the EventScheduler.
     */
    Result<TaskId> scheduleTask(TaskParams taskParams);

    /**
     * Cancels the pending task with given TaskId in the IEventScheduler (and does not run the task).
     * Should be called when the IEventScheduler is in the Running state.
     *
     * @param[in] taskId The task to be cancelled in IEventScheduler
     * @return
     *   - TTV_EC_SUCCESS: The task was successfully cancelled.
     *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has not yet been initialized or has already been shutdown.
     *   - TTV_EC_OPERATION_FAILED: The task was not found in the IEventScheduler and was failed to be cancelled.
     */
    ErrorCode cancelTask(TaskId taskId);

    /**
     * Asynchronously shutdown the IEventScheduler.
     * This must be called before IEventScheduler is destroyed.
     * All tasks in the IEventScheduler should be cancelled in the normal task execution context for the
     * IEventScheduler. Should be called when the IEventScheduler is in the Running state. After calling Shutdown, the
     * IEventScheduler will move to the ShuttingDown state and then eventually the ShutDown state once shutdownTask is
     * fired.
     *
     * @param[in] shutdownTask The task to be called by the IEventScheduler once the IEventScheduler is shutdown.
     * @return
     *   - TTV_EC_SUCCESS: The request to Shutdown the IEventScheduler was sent successfully.
     *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has already been shutdown.
     */
    ErrorCode shutdown(TaskFunction shutdownCallback);

    /**
     * Returns the state of the IEventScheduler.
     */
    EventSchedulerState getState();

    /**
     * Invalidates the object and releases all associated native resources.
     */
    void dispose();
}
