package tv.twitch;

/**
 * Before using any feature of the Twitch SDK the native library must first
 * be loaded and initialized.
 */
public class Library {
    /**
     * @Deprecated use {@link Library(IJniThreadChecker)}
     */
    @Deprecated
    public Library() {
        this.sJniThreadValidator = PassThroughJniThreadValidator.INSTANCE;
    }

    public Library(IJniThreadChecker jniThreadChecker) {
        if (jniThreadChecker == null) {
            throw new IllegalArgumentException("jniThreadChecker must not be null");
        }
        this.sJniThreadValidator = new JniThreadValidator(jniThreadChecker);
    }

    protected native ErrorCode Initialize();
    protected native ErrorCode Shutdown();
    protected native String GetVersionString();

    protected native ErrorCode SetClientId(String clientId);
    protected native ErrorCode SetComponentMessageLevel(String component, /*MessageLevel*/ int traceLevel);
    protected native ErrorCode SetGlobalMessageLevel(/*MessageLevel*/ int traceLevel);
    protected native String ErrorToString(ErrorCode err);
    protected native ErrorCode SetHttpRequestProvider(IHttpRequestProvider http);
    protected native ErrorCode SetEventTracker(IEventTracker tracker);
    protected native ErrorCode RegisterSocketFactory(ISocketFactory factory);
    protected native ErrorCode UnregisterSocketFactory(ISocketFactory factory);
    protected native ErrorCode RegisterWebSocketFactory(IWebSocketFactory factory);
    protected native ErrorCode UnregisterWebSocketFactory(IWebSocketFactory factory);
    // SDK-754 this API doesn't work, but it is unused on Android
    protected native Result<IEventScheduler> CreateBackgroundEventScheduler(IJniThreadValidator jniThreadValidator);
    protected native void SetTracer(ITracer tracer);
    protected native long GetSystemClockTime();
    protected native long MsToSystemTime(long milliseconds);

    protected boolean sInitialized = false;

    private final IJniThreadValidator sJniThreadValidator;

    public boolean isInitialized() { return sInitialized; }

    /**
     * Performs the loading of the native library. Returns TTV_EC_SUCCESS if the load was successful.
     * Override this method if you have an alternate means of loading the library.
     */
    public ErrorCode loadLibrary(String libraryName) {
        final String libraryName_;
        if (libraryName == null) {
            libraryName_ = "twitchsdk";
        } else {
            libraryName_ = libraryName;
        }

        try {
            sJniThreadValidator.callJniRunnable(new Runnable() {
                @Override
                public void run() {
                    System.loadLibrary(libraryName_);
                }
            });
        } catch (UnsatisfiedLinkError e) {
            System.err.println("If on Windows, make sure to provide all of the necessary "
                + "dll's as specified in the twitchsdk README. Also, make sure to set the PATH "
                + "environment variable to point to the directory containing the dll's.");
            throw e;
        }

        return CoreErrorCode.TTV_EC_SUCCESS;
    }

    public ErrorCode initialize() { return initialize(null); }

    public ErrorCode initialize(String libraryName) {
        if (sInitialized) {
            return CoreErrorCode.TTV_EC_ALREADY_INITIALIZED;
        }

        ErrorCode ec = loadLibrary(libraryName);

        if (ec.failed()) {
            return ec;
        }

        try {
            ec = sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return Initialize();
                }
            });

            if (ec.succeeded()) {
                sInitialized = true;
            }
        } catch (UnsatisfiedLinkError ex) {
            System.err.println("Could not find the JNI entrypoint Library_Initialize().");
            throw ex;
        }

        return ec;
    }

    public ErrorCode setClientId(final String clientId) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetClientId(clientId);
            }
        });
    }

    public ErrorCode setComponentMessageLevel(final String component, MessageLevel traceLevel) {
        final MessageLevel traceLevel_;
        if (traceLevel == null) {
            traceLevel_ = MessageLevel.TTV_ML_NONE;
        } else {
            traceLevel_ = traceLevel;
        }

        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetComponentMessageLevel(component, traceLevel_.getValue());
            }
        });
    }

    public ErrorCode setGlobalMessageLevel(MessageLevel traceLevel) {
        final MessageLevel traceLevel_;
        if (traceLevel == null) {
            traceLevel_ = MessageLevel.TTV_ML_NONE;
        } else {
            traceLevel_ = traceLevel;
        }

        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetGlobalMessageLevel(traceLevel_.getValue());
            }
        });
    }

    public String errorToString(final ErrorCode err) {
        if (err == null) {
            return null;
        }

        return sJniThreadValidator.callJniCallable(new IJniCallable<String>() {
            @Override
            public String call() {
                return ErrorToString(err);
            }
        });
    }

    public void setTracer(final ITracer tracer) {
        sJniThreadValidator.callJniRunnable(new Runnable() {
            @Override
            public void run() {
                SetTracer(tracer);
            }
        });
    }

    public ErrorCode setHttpRequestProvider(final IHttpRequestProvider http) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetHttpRequestProvider(http);
            }
        });
    }

    public ErrorCode setEventTracker(final IEventTracker tracker) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetEventTracker(tracker);
            }
        });
    }

    public ErrorCode registerSocketFactory(final ISocketFactory factory) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RegisterSocketFactory(factory);
            }
        });
    }

    public ErrorCode unregisterSocketFactory(final ISocketFactory factory) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UnregisterSocketFactory(factory);
            }
        });
    }

    public ErrorCode registerWebSocketFactory(final IWebSocketFactory factory) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RegisterWebSocketFactory(factory);
            }
        });
    }

    public ErrorCode unregisterWebSocketFactory(final IWebSocketFactory factory) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UnregisterWebSocketFactory(factory);
            }
        });
    }

    @Deprecated
    public Result<IEventScheduler> createBackgroundEventScheduler() {
        // SDK-754 this API doesn't work, but it is unused on Android
        return sJniThreadValidator.callJniCallable(new IJniCallable<Result<IEventScheduler>>() {
            @Override
            public Result<IEventScheduler> call() {
                return CreateBackgroundEventScheduler(sJniThreadValidator);
            }
        });
    }

    public long getSystemClockTime() {
        return sJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return GetSystemClockTime();
            }
        });
    }

    public long msToSystemTime(final long milliseconds) {
        return sJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return MsToSystemTime(milliseconds);
            }
        });
    }

    public ErrorCode shutdown() {
        if (!sInitialized) {
            return CoreErrorCode.TTV_EC_NOT_INITIALIZED;
        }

        ErrorCode ec = sJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Shutdown();
            }
        });

        if (ec.succeeded()) {
            sInitialized = false;
        }

        return ec;
    }
}
