/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/channel/ichannelstatus.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/generated/java_all.h"
#include "twitchsdk/core/generated/java_ichannelstatus.h"
#include "twitchsdk/core/generated/java_idashboardactivitystatus.h"
#include "twitchsdk/core/generated/java_igenericsubscriberstatus.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/generated/jni_coreapi.h"
#include "twitchsdk/core/idashboardactivitystatus.h"
#include "twitchsdk/core/igenericsubscriberstatus.h"
#include "twitchsdk/core/java_channellistenerproxy.h"
#include "twitchsdk/core/java_coreapilistenerproxy.h"
#include "twitchsdk/core/java_coreutil.h"
#include "twitchsdk/core/java_dashboardactivitylistenerproxy.h"
#include "twitchsdk/core/java_genericsubscriberlistenerproxy.h"
#include "twitchsdk/core/java_httprequest.h"

using namespace ttv;
using namespace ttv::binding::java;

#define GET_COREAPI_PTR(x) reinterpret_cast<CoreAPI*>(x)

namespace ttv {
TTV_ErrorCode SetClientId(const std::string& clientId);
}

std::shared_ptr<ttv::CoreAPI> ttv::binding::java::GetCoreApiInstance(jobject jCoreApi) {
  return gCoreApiNativeProxyRegistry.LookupNativeInstance(jCoreApi);
}

JNIEXPORT jlong JNICALL Java_tv_twitch_CoreAPI_CreateNativeInstance(JNIEnv* jEnv, jobject jThis) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  // Force load all types to ensure all classes are loaded by the time we need them.
  LoadAllCoreJavaClassInfo(jEnv);

  std::shared_ptr<CoreApiContext> context = std::make_shared<CoreApiContext>();
  context->coreApi = std::make_shared<CoreAPI>();
  context->nativeListener = std::make_shared<JavaCoreAPIListenerProxy>(jThis);

  gCoreApiNativeProxyRegistry.Register(context->coreApi, context, jThis);

  context->coreApi->SetListener(context->nativeListener);

  return reinterpret_cast<jlong>(context->coreApi.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_CoreAPI_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jNativePointer) {
  gCoreApiNativeProxyRegistry.Unregister(jNativePointer);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_GetState(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  IModule::State state = api->GetState();

  JavaClassInfo& info = GetJavaClassInfo_ModuleState(jEnv);
  return GetJavaInstance_SimpleEnum(jEnv, info, state);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_CoreAPI_GetModuleName(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  std::string name = api->GetModuleName();

  return GetJavaInstance_String(jEnv, name.c_str());
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_Initialize(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_InitializeCallback(jEnv));

  TTV_ErrorCode ec = api->Initialize([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_ShutdownCallback(jEnv));

  TTV_ErrorCode ec = api->Shutdown([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_Update(JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto keepAlive = gCoreApiNativeProxyRegistry.LookupNativeInstance(jThis);

  TTV_ErrorCode ec = api->Update();
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_SetListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gCoreApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    context->nativeListener->SetListener(jListener);
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_GetRequiredOAuthScopes(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jModuleNamesResultContainer, jobject jScopesResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jModuleNamesResultContainer, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jScopesResultContainer, TTV_EC_INVALID_ARG);

  std::vector<std::string> modules;
  std::vector<std::string> allScopes;

  TTV_ErrorCode ec = api->GetRequiredOAuthScopes(modules, allScopes);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jModules, GetJavaInstance_StringArray(jEnv, modules));
    AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jAllScopes, GetJavaInstance_StringArray(jEnv, allScopes));

    SetResultContainerResult(jEnv, jModuleNamesResultContainer, jModules);
    SetResultContainerResult(jEnv, jScopesResultContainer, jAllScopes);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_LogIn(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jOAuthToken, jobject jCallBack) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jOAuthToken, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter oauthToken(jEnv, jOAuthToken);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallBack);

  TTV_ErrorCode ec = api->LogIn(
    oauthToken.GetNativeString(), [callbackReference](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_LogInCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jErrorCode,
          GetJavaInstance_ErrorCode(gActiveJavaEnvironment, errorDetails.ec));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jUserInfo,
          ((TTV_SUCCEEDED(errorDetails.ec)) ? GetJavaInstance_UserInfo(gActiveJavaEnvironment, userInfo) : nullptr));
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jUserInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_LogOut(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->LogOut(static_cast<UserId>(jUserId), [callbackReference](TTV_ErrorCode callbackEc) {
    auto callback = callbackReference->GetInstance();
    if (callback != nullptr) {
      auto callbackClassInfo = GetJavaClassInfo_CoreAPI_LogOutCallback(gActiveJavaEnvironment);
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode);
    }
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_ConnectPubSub(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = api->ConnectPubSub(static_cast<UserId>(jUserId));
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_DisconnectPubSub(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = api->DisconnectPubSub(static_cast<UserId>(jUserId));
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_SetGlobalSetting(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jKey, jstring jValue) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_ErrorCode ec;
  if (jKey != nullptr && jValue != nullptr) {
    ScopedJavaUTFStringConverter key(jEnv, jKey);
    ScopedJavaUTFStringConverter value(jEnv, jValue);

    ec = api->SetGlobalSetting(key.GetNativeString(), value.GetNativeString());
  } else {
    ec = TTV_EC_INVALID_ARG;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_RemoveGlobalSetting(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jKey) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  if (jKey == nullptr) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  ScopedJavaUTFStringConverter key(jEnv, jKey);

  TTV_ErrorCode ec = api->RemoveGlobalSetting(key.GetNativeString());
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_GetGlobalSetting(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jKey, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  if (jKey == nullptr || jResultContainer == nullptr) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  ScopedJavaUTFStringConverter key(jEnv, jKey);

  std::string value;
  TTV_ErrorCode ec = api->GetGlobalSetting(key.GetNativeString(), value);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jValue, GetJavaInstance_String(jEnv, value));
    SetResultContainerResult(jEnv, jResultContainer, jValue);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_GetSubscribedPubsubTopics(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jTopicsResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jTopicsResultContainer, TTV_EC_INVALID_ARG);

  std::vector<std::string> topics;

  TTV_ErrorCode ec = api->GetSubscribedPubsubTopics(topics);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jTopics, GetJavaInstance_StringArray(jEnv, topics));

    SetResultContainerResult(jEnv, jTopicsResultContainer, jTopics);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT void JNICALL Java_tv_twitch_CoreAPI_CrashAbort(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  api->CrashAbort();
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_SetLocalLanguage(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jLanguage) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jLanguage, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter language(jEnv, jLanguage);

  TTV_ErrorCode ec = api->SetLocalLanguage(language.GetNativeString());
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_GetLocalLanguage(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  std::string language;
  TTV_ErrorCode ec = api->GetLocalLanguage(language);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jString, GetJavaInstance_String(jEnv, language));
    SetResultContainerResult(jEnv, jResultContainer, jString);
  } else {
    SetResultContainerResult(jEnv, jResultContainer, nullptr);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchUserInfoById(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchUserInfoById(
    static_cast<UserId>(jUserId), [callbackReference](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchUserInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jErrorCode,
          GetJavaInstance_ErrorCode(gActiveJavaEnvironment, errorDetails.ec));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jUserInfo,
          ((TTV_SUCCEEDED(errorDetails.ec)) ? GetJavaInstance_UserInfo(gActiveJavaEnvironment, userInfo) : nullptr));
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jUserInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchUserInfoByName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jUserName, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter userName(jEnv, jUserName);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchUserInfoByName(
    userName.GetNativeString(), [callbackReference](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchUserInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jErrorCode,
          GetJavaInstance_ErrorCode(gActiveJavaEnvironment, errorDetails.ec));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jUserInfo,
          ((TTV_SUCCEEDED(errorDetails.ec)) ? GetJavaInstance_UserInfo(gActiveJavaEnvironment, userInfo) : nullptr))
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jUserInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchChannelInfoById(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jChannelId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchChannelInfoById(
    static_cast<ChannelId>(jChannelId), [callbackReference](TTV_ErrorCode callbackEc, const ChannelInfo& channelInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchChannelInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jChannelInfo,
          ((TTV_SUCCEEDED(callbackEc)) ? GetJavaInstance_ChannelInfo(gActiveJavaEnvironment, channelInfo) : nullptr));
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jChannelInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchChannelInfoByName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jChannelName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jChannelName, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter channelName(jEnv, jChannelName);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchChannelInfoByName(
    channelName.GetNativeString(), [callbackReference](TTV_ErrorCode callbackEc, const ChannelInfo& channelInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchChannelInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jChannelInfo,
          ((TTV_SUCCEEDED(callbackEc)) ? GetJavaInstance_ChannelInfo(gActiveJavaEnvironment, channelInfo) : nullptr))
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jChannelInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchStreamInfoById(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jChannelId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchStreamInfoById(
    static_cast<ChannelId>(jChannelId), [callbackReference](TTV_ErrorCode callbackEc, const StreamInfo& streamInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchStreamInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jStreamInfo,
          ((TTV_SUCCEEDED(callbackEc)) ? GetJavaInstance_StreamInfo(gActiveJavaEnvironment, streamInfo) : nullptr));
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jStreamInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_FetchStreamInfoByName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jChannelName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jChannelName, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter channelName(jEnv, jChannelName);

  auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
  callbackReference->Bind(jEnv, jCallback);

  TTV_ErrorCode ec = api->FetchStreamInfoByName(
    channelName.GetNativeString(), [callbackReference](TTV_ErrorCode callbackEc, const StreamInfo& streamInfo) {
      auto callback = callbackReference->GetInstance();
      if (callback != nullptr) {
        auto callbackClassInfo = GetJavaClassInfo_CoreAPI_FetchStreamInfoCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jStreamInfo,
          ((TTV_SUCCEEDED(callbackEc)) ? GetJavaInstance_StreamInfo(gActiveJavaEnvironment, streamInfo) : nullptr))
        gActiveJavaEnvironment->CallVoidMethod(callback, callbackClassInfo.methods["invoke"], jErrorCode, jStreamInfo);
      }
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_CreateChannelStatus(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jJniThreadValidator, jint jUserId, jint jChannelId, jobject jListener,
  jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChannelStatus> result;

  auto context = gCoreApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaChannelListenerProxy> listenerProxy = std::make_shared<JavaChannelListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec =
      api->CreateChannelStatus(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChannelStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChannelStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()), jJniThreadValidator));
    SetResultContainerResult(jEnv, jResultContainer, jChannelStatus);

    gIChannelStatusInstanceRegistry.Register(result, context, jChannelStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_DisposeChannelStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jChannelStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIChannelStatusInstanceRegistry.LookupNativeInstance(jChannelStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIChannelStatusInstanceRegistry.Unregister(jChannelStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_CreateDashboardActivityStatus(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IDashboardActivityStatus> result;

  auto context = gCoreApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaDashboardActivityListenerProxy> listenerProxy =
      std::make_shared<JavaDashboardActivityListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateDashboardActivityStatus(
      static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_DashboardActivityStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jDashboardActivityStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jDashboardActivityStatus);

    gIDashboardActivityStatusInstanceRegistry.Register(result, context, jDashboardActivityStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_DisposeDashboardActivityStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jDashboardActivityStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIDashboardActivityStatusInstanceRegistry.LookupNativeInstance(jDashboardActivityStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIDashboardActivityStatusInstanceRegistry.Unregister(jDashboardActivityStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_CreateGenericSubscriberStatus(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jstring jTopic, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_COREAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IGenericSubscriberStatus> result;

  auto context = gCoreApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaGenericSubscriberListenerProxy> listenerProxy =
      std::make_shared<JavaGenericSubscriberListenerProxy>();
    listenerProxy->SetListener(jListener);

    ScopedJavaUTFStringConverter topic(jEnv, jTopic);
    ec =
      api->CreateGenericSubscriberStatus(static_cast<UserId>(jUserId), topic.GetNativeString(), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_GenericSubscriberStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jGenericSubscriberStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jGenericSubscriberStatus);

    gIGenericSubscriberStatusInstanceRegistry.Register(result, context, jGenericSubscriberStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_CoreAPI_DisposeGenericSubscriberStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jGenericSubscriberStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIGenericSubscriberStatusInstanceRegistry.LookupNativeInstance(jGenericSubscriberStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIGenericSubscriberStatusInstanceRegistry.Unregister(jGenericSubscriberStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
