/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/java_coreutil.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/generated/java_all.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/utf8.h"

#include <stdlib.h>

#include <functional>
#include <iostream>

using namespace ttv;

ttv::binding::java::JavaNativeProxyRegistry<ttv::CoreAPI, ttv::binding::java::CoreApiContext>
  ttv::binding::java::gCoreApiNativeProxyRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ttv::IChannelStatus, ttv::binding::java::CoreApiContext>
  ttv::binding::java::gIChannelStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ttv::IDashboardActivityStatus, ttv::binding::java::CoreApiContext>
  ttv::binding::java::gIDashboardActivityStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ttv::IGenericSubscriberStatus, ttv::binding::java::CoreApiContext>
  ttv::binding::java::gIGenericSubscriberStatusInstanceRegistry;

void ttv::binding::java::LoadAllCoreJavaClassInfo(JNIEnv* jEnv) {
  GetJavaClassInfo_IModule_InitializeCallback(jEnv);
  GetJavaClassInfo_IModule_ShutdownCallback(jEnv);

  GetJavaClassInfo_CoreAPI_LogInCallback(jEnv);
  GetJavaClassInfo_CoreAPI_LogOutCallback(jEnv);
  GetJavaClassInfo_CoreAPI_FetchUserInfoCallback(jEnv);
  GetJavaClassInfo_CoreAPI_FetchChannelInfoCallback(jEnv);
  GetJavaClassInfo_CoreAPI_FetchStreamInfoCallback(jEnv);

  GetJavaClassInfo_AuthToken(jEnv);
  GetJavaClassInfo_CanTheyError(jEnv);
  GetJavaClassInfo_ChannelInfo(jEnv);
  GetJavaClassInfo_ChannelStatusProxy(jEnv);
  GetJavaClassInfo_CorePubSubState(jEnv);
  GetJavaClassInfo_EventSchedulerProxy(jEnv);
  GetJavaClassInfo_EventSchedulerState(jEnv);
  GetJavaClassInfo_IChannelListener(jEnv);
  GetJavaClassInfo_IChannelStatus(jEnv);
  GetJavaClassInfo_ICoreAPIListener(jEnv);
  GetJavaClassInfo_IEventScheduler(jEnv);
  GetJavaClassInfo_IEventTracker(jEnv);
  GetJavaClassInfo_IModule(jEnv);
  GetJavaClassInfo_IModuleListener(jEnv);
  GetJavaClassInfo_ITracer(jEnv);
  GetJavaClassInfo_ModuleState(jEnv);
  GetJavaClassInfo_PreviewImages(jEnv);
  GetJavaClassInfo_ProfileImage(jEnv);
  GetJavaClassInfo_SquadInfo(jEnv);
  GetJavaClassInfo_SquadMember(jEnv);
  GetJavaClassInfo_SquadStatus(jEnv);
  GetJavaClassInfo_StreamInfo(jEnv);
  GetJavaClassInfo_StreamInfoUpdate(jEnv);
  GetJavaClassInfo_StreamType(jEnv);
  GetJavaClassInfo_UserInfo(jEnv);
  GetJavaClassInfo_VodType(jEnv);
  GetJavaClassInfo_WatchPartyUpdate(jEnv);

  GetJavaClassInfo_IGenericSubscriberListener(jEnv);
  GetJavaClassInfo_IGenericSubscriberStatus(jEnv);
  GetJavaClassInfo_GenericSubscriberStatusProxy(jEnv);

  GetJavaClassInfo_IDashboardActivityListener(jEnv);
  GetJavaClassInfo_IDashboardActivityStatus(jEnv);
  GetJavaClassInfo_DashboardActivityStatusProxy(jEnv);

  GetJavaClassInfo_DashboardActivityHeader(jEnv);
  GetJavaClassInfo_DashboardActivityUser(jEnv);
  GetJavaClassInfo_DashboardActivityFragment(jEnv);
  GetJavaClassInfo_DashboardActivityBitsUsage(jEnv);
  GetJavaClassInfo_DashboardActivityFollow(jEnv);
  GetJavaClassInfo_DashboardActivityHost(jEnv);
  GetJavaClassInfo_DashboardActivityRaiding(jEnv);
  GetJavaClassInfo_DashboardActivitySubscription(jEnv);
  GetJavaClassInfo_DashboardActivityResubscriptionSharing(jEnv);
  GetJavaClassInfo_DashboardActivitySubscriptionGiftingCommunity(jEnv);
  GetJavaClassInfo_DashboardActivitySubscriptionGiftingIndividual(jEnv);
}

jobject ttv::binding::java::GetJavaInstance_PubSubState(JNIEnv* jEnv, PubSubState state) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_CorePubSubState(jEnv);
  return jEnv->CallStaticObjectMethod(info.klass, info.staticMethods["lookupValue"], static_cast<jint>(state));
}

jobject ttv::binding::java::GetJavaInstance_UserInfo(JNIEnv* jEnv, const UserInfo& userInfo) {
  JavaClassInfo& info = GetJavaClassInfo_UserInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["userId"], static_cast<jint>(userInfo.userId));
  jEnv->SetIntField(jInstance, info.fields["createdTimestamp"], static_cast<jint>(userInfo.createdTimestamp));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, userInfo.userName));
  jEnv->SetObjectField(jInstance, info.fields["userName"], jUserName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, userInfo.displayName));
  jEnv->SetObjectField(jInstance, info.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jBio, GetJavaInstance_String(jEnv, userInfo.bio));
  jEnv->SetObjectField(jInstance, info.fields["bio"], jBio);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLogoImageUrl, GetJavaInstance_String(jEnv, userInfo.logoImageUrl));
  jEnv->SetObjectField(jInstance, info.fields["logoImageUrl"], jLogoImageUrl);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_UserInfoArray(JNIEnv* jEnv, const std::vector<UserInfo>& list) {
  JavaClassInfo& info = GetJavaClassInfo_UserInfo(jEnv);

  jobjectArray jArray = GetJavaInstance_Array(jEnv, info, static_cast<uint32_t>(list.size()),
    [jEnv, &list](uint32_t index) { return GetJavaInstance_UserInfo(jEnv, list[index]); });

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_CanTheyError(JNIEnv* jEnv, const ttv::CanTheyError& canTheyError) {
  JavaClassInfo& info = GetJavaClassInfo_CanTheyError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jCode, GetJavaInstance_String(jEnv, canTheyError.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jCode);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessage, GetJavaInstance_String(jEnv, canTheyError.message));
  jEnv->SetObjectField(jInstance, info.fields["message"], jMessage);

  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jLinks, GetJavaInstance_StringArray(jEnv, canTheyError.links));
  jEnv->SetObjectField(jInstance, info.fields["links"], jLinks);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChannelInfo(JNIEnv* jEnv, const ttv::ChannelInfo& channelInfo) {
  JavaClassInfo& info = GetJavaClassInfo_ChannelInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, channelInfo.displayName));
  jEnv->SetObjectField(jInstance, info.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, channelInfo.name));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGame, GetJavaInstance_String(jEnv, channelInfo.game));
  jEnv->SetObjectField(jInstance, info.fields["game"], jGame);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDescription, GetJavaInstance_String(jEnv, channelInfo.description));
  jEnv->SetObjectField(jInstance, info.fields["description"], jDescription);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jStatus, GetJavaInstance_String(jEnv, channelInfo.status));
  jEnv->SetObjectField(jInstance, info.fields["status"], jStatus);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLanguage, GetJavaInstance_String(jEnv, channelInfo.language));
  jEnv->SetObjectField(jInstance, info.fields["language"], jLanguage);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jBroadcasterLanguage, GetJavaInstance_String(jEnv, channelInfo.broadcasterLanguage));
  jEnv->SetObjectField(jInstance, info.fields["broadcasterLanguage"], jBroadcasterLanguage);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLogoImageUrl, GetJavaInstance_String(jEnv, channelInfo.logoImageUrl));
  jEnv->SetObjectField(jInstance, info.fields["logoImageUrl"], jLogoImageUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jChannelUrl, GetJavaInstance_String(jEnv, channelInfo.channelUrl));
  jEnv->SetObjectField(jInstance, info.fields["channelUrl"], jChannelUrl);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jVideoBannerImageUrl, GetJavaInstance_String(jEnv, channelInfo.videoBannerImageUrl));
  jEnv->SetObjectField(jInstance, info.fields["videoBannerImageUrl"], jVideoBannerImageUrl);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jProfileBannerImageUrl, GetJavaInstance_String(jEnv, channelInfo.profileBannerImageUrl));
  jEnv->SetObjectField(jInstance, info.fields["profileBannerImageUrl"], jProfileBannerImageUrl);

  jEnv->SetIntField(jInstance, info.fields["channelId"], static_cast<jint>(channelInfo.channelId));
  jEnv->SetIntField(jInstance, info.fields["createdAtTimestamp"], static_cast<jint>(channelInfo.createdAtTimestamp));
  jEnv->SetIntField(jInstance, info.fields["updatedAtTimestamp"], static_cast<jint>(channelInfo.updatedAtTimestamp));
  jEnv->SetIntField(jInstance, info.fields["numFollowers"], static_cast<jint>(channelInfo.numFollowers));
  jEnv->SetIntField(jInstance, info.fields["numViews"], static_cast<jint>(channelInfo.numViews));

  jEnv->SetBooleanField(jInstance, info.fields["mature"], channelInfo.mature ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["partner"], channelInfo.partner ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["affiliate"], channelInfo.affiliate ? JNI_TRUE : JNI_FALSE);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_WatchPartyUpdate(JNIEnv* jEnv, const ttv::WatchPartyUpdate& update) {
  JavaClassInfo& info = GetJavaClassInfo_WatchPartyUpdate(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jIncrementUrl, GetJavaInstance_String(jEnv, update.incrementUrl));
  jEnv->SetObjectField(jInstance, info.fields["incrementUrl"], jIncrementUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jVodId, GetJavaInstance_String(jEnv, update.vodId));
  jEnv->SetObjectField(jInstance, info.fields["vodId"], jVodId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTitle, GetJavaInstance_String(jEnv, update.title));
  jEnv->SetObjectField(jInstance, info.fields["title"], jTitle);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jWatchPartyId, GetJavaInstance_String(jEnv, update.watchPartyId));
  jEnv->SetObjectField(jInstance, info.fields["watchPartyId"], jWatchPartyId);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBroadcastType,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_VodType(jEnv), update.broadcastType));
  jEnv->SetObjectField(jInstance, info.fields["broadcastType"], jBroadcastType);

  jEnv->SetBooleanField(jInstance, info.fields["viewable"], (update.viewable ? JNI_TRUE : JNI_FALSE));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SquadMember(JNIEnv* jEnv, const SquadMember& member) {
  JavaClassInfo& info = GetJavaClassInfo_SquadMember(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLogin, GetJavaInstance_String(jEnv, member.userLogin));
  jEnv->SetObjectField(jInstance, info.fields["userLogin"], jLogin);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, member.userDisplayName));
  jEnv->SetObjectField(jInstance, info.fields["userDisplayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jProfileImage, GetJavaInstance_String(jEnv, member.profileImageUrl150));
  jEnv->SetObjectField(jInstance, info.fields["profileImageUrl150"], jProfileImage);

  jEnv->SetIntField(jInstance, info.fields["channelId"], static_cast<jint>(member.channelId));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SquadInfo(JNIEnv* jEnv, const SquadInfo& squad) {
  JavaClassInfo& info = GetJavaClassInfo_SquadInfo(jEnv);
  JavaClassInfo& squadMemberInfo = GetJavaClassInfo_SquadMember(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSquadId, GetJavaInstance_String(jEnv, squad.squadId));
  jEnv->SetObjectField(jInstance, info.fields["squadId"], jSquadId);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jStatus, GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_SquadStatus(jEnv), squad.status));
  jEnv->SetObjectField(jInstance, info.fields["status"], jStatus);

  jEnv->SetIntField(jInstance, info.fields["ownerId"], static_cast<jint>(squad.ownerId));

  // Create the squad member array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jMembers,
    jEnv->NewObjectArray(static_cast<jsize>(squad.members.size()), squadMemberInfo.klass, nullptr));

  jsize index = 0;
  for (const auto& member : squad.members) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMember, GetJavaInstance_SquadMember(jEnv, member));
    jEnv->SetObjectArrayElement(jMembers, index, jMember);
    index++;
  }

  jEnv->SetObjectField(jInstance, info.fields["members"], jMembers);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_PreviewImages(JNIEnv* jEnv, const PreviewImages& previewImages) {
  JavaClassInfo& info = GetJavaClassInfo_PreviewImages(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLargeUrl, GetJavaInstance_String(jEnv, previewImages.largeUrl));
  jEnv->SetObjectField(jInstance, info.fields["largeUrl"], jLargeUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMediumUrl, GetJavaInstance_String(jEnv, previewImages.mediumUrl));
  jEnv->SetObjectField(jInstance, info.fields["mediumUrl"], jMediumUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSmallUrl, GetJavaInstance_String(jEnv, previewImages.smallUrl));
  jEnv->SetObjectField(jInstance, info.fields["smallUrl"], jSmallUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTemplateUrl, GetJavaInstance_String(jEnv, previewImages.templateUrl));
  jEnv->SetObjectField(jInstance, info.fields["templateUrl"], jTemplateUrl);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_StreamInfo(JNIEnv* jEnv, const ttv::StreamInfo& streamInfo) {
  JavaClassInfo& info = GetJavaClassInfo_StreamInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChannelInfo, GetJavaInstance_ChannelInfo(jEnv, streamInfo.channelInfo));
  jEnv->SetObjectField(jInstance, info.fields["channelInfo"], jChannelInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPreviewImages, GetJavaInstance_PreviewImages(jEnv, streamInfo.previewImages));
  jEnv->SetObjectField(jInstance, info.fields["previewImages"], jPreviewImages);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGame, GetJavaInstance_String(jEnv, streamInfo.game));
  jEnv->SetObjectField(jInstance, info.fields["game"], jGame);

  jEnv->SetDoubleField(jInstance, info.fields["averageFPS"], static_cast<jdouble>(streamInfo.averageFPS));

  jEnv->SetLongField(jInstance, info.fields["streamId"], static_cast<jlong>(streamInfo.streamId));
  jEnv->SetLongField(jInstance, info.fields["delay"], static_cast<jlong>(streamInfo.delay));
  jEnv->SetLongField(jInstance, info.fields["viewerCount"], static_cast<jlong>(streamInfo.viewerCount));

  jEnv->SetLongField(jInstance, info.fields["archiveVideoId"], static_cast<jlong>(streamInfo.archiveVideoId));

  jEnv->SetIntField(jInstance, info.fields["videoHeight"], static_cast<jint>(streamInfo.videoHeight));
  jEnv->SetIntField(jInstance, info.fields["createdAtTimestamp"], static_cast<jint>(streamInfo.createdAtTimestamp));

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBroadcastPlatform,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_BroadcastPlatform(jEnv), streamInfo.broadcastPlatform));
  jEnv->SetObjectField(jInstance, info.fields["broadcastPlatform"], jBroadcastPlatform);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jStreamType,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_StreamType(jEnv), streamInfo.streamType));
  jEnv->SetObjectField(jInstance, info.fields["streamType"], jStreamType);

  jEnv->SetBooleanField(jInstance, info.fields["isPlaylist"], (streamInfo.isPlaylist ? JNI_TRUE : JNI_FALSE));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_StreamInfoUpdate(
  JNIEnv* jEnv, const ttv::StreamInfoUpdate& streamInfoUpdate) {
  JavaClassInfo& info = GetJavaClassInfo_StreamInfoUpdate(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTitle, GetJavaInstance_String(jEnv, streamInfoUpdate.title));
  jEnv->SetObjectField(jInstance, info.fields["title"], jTitle);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGame, GetJavaInstance_String(jEnv, streamInfoUpdate.game));
  jEnv->SetObjectField(jInstance, info.fields["game"], jGame);

  jEnv->SetIntField(jInstance, info.fields["gameId"], static_cast<jint>(streamInfoUpdate.gameId));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ProfileImage(JNIEnv* jEnv, const ttv::ProfileImage& image) {
  JavaClassInfo& info = GetJavaClassInfo_ProfileImage(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUrl, GetJavaInstance_String(jEnv, image.url));
  jEnv->SetObjectField(jInstance, info.fields["url"], jUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jFormat, GetJavaInstance_String(jEnv, image.format));
  jEnv->SetObjectField(jInstance, info.fields["format"], jFormat);

  jEnv->SetIntField(jInstance, info.fields["width"], static_cast<jint>(image.width));
  jEnv->SetIntField(jInstance, info.fields["height"], static_cast<jint>(image.height));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ProfileImageArray(
  JNIEnv* jEnv, const std::vector<ttv::ProfileImage>& images) {
  JavaClassInfo& socialFriendInfo = GetJavaClassInfo_ProfileImage(jEnv);

  jobjectArray jArray = GetJavaInstance_Array(jEnv, socialFriendInfo, static_cast<uint32_t>(images.size()),
    [jEnv, &images](uint32_t index) { return GetJavaInstance_ProfileImage(jEnv, images[index]); });

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_TrackingValue(JNIEnv* jEnv, const TrackingValue& value) {
  switch (value.GetType()) {
    case TrackingValue::Type::Null:
      return nullptr;
    case TrackingValue::Type::Boolean:
      return GetJavaInstance_Boolean(jEnv, value.GetBooleanValue());
    case TrackingValue::Type::Integer:
      return GetJavaInstance_Long(jEnv, value.GetIntegerValue());
    case TrackingValue::Type::Double:
      return GetJavaInstance_Double(jEnv, value.GetDoubleValue());
    case TrackingValue::Type::String:
      return GetJavaInstance_String(jEnv, value.GetStringValue());
    default:
      return nullptr;
  }
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityHeader(JNIEnv* jEnv, const DashboardActivityHeader& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityHeader(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jId, GetJavaInstance_String(jEnv, data.id));
  jEnv->SetObjectField(jInstance, info.fields["id"], jId);

  jEnv->SetIntField(jInstance, info.fields["timestamp"], static_cast<jint>(data.timestamp));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityUser(JNIEnv* jEnv, const DashboardActivityUser& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityUser(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, data.displayName));
  jEnv->SetObjectField(jInstance, info.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLogin, GetJavaInstance_String(jEnv, data.login));
  jEnv->SetObjectField(jInstance, info.fields["login"], jLogin);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserId, GetJavaInstance_String(jEnv, data.userId));
  jEnv->SetObjectField(jInstance, info.fields["userId"], jUserId);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityFragment(
  JNIEnv* jEnv, const DashboardActivityFragment& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityFragment(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonId, GetJavaInstance_String(jEnv, data.emoticonId));
  jEnv->SetObjectField(jInstance, info.fields["emoticonId"], jEmoticonId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonSetId, GetJavaInstance_String(jEnv, data.emoticonSetId));
  jEnv->SetObjectField(jInstance, info.fields["emoticonSetId"], jEmoticonSetId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jText, GetJavaInstance_String(jEnv, data.text));
  jEnv->SetObjectField(jInstance, info.fields["text"], jText);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityBitsUsage(
  JNIEnv* jEnv, const DashboardActivityBitsUsage& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityBitsUsage(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUser, GetJavaInstance_DashboardActivityUser(jEnv, data.user));
  jEnv->SetObjectField(jInstance, info.fields["user"], jUser);

  jEnv->SetIntField(jInstance, info.fields["amount"], static_cast<jint>(data.amount));
  jEnv->SetBooleanField(jInstance, info.fields["anonymous"], (data.anonymous ? JNI_TRUE : JNI_FALSE));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityFollow(JNIEnv* jEnv, const DashboardActivityFollow& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityFollow(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jFollower, GetJavaInstance_DashboardActivityUser(jEnv, data.follower));
  jEnv->SetObjectField(jInstance, info.fields["follower"], jFollower);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityHost(JNIEnv* jEnv, const DashboardActivityHost& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityHost(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHost, GetJavaInstance_DashboardActivityUser(jEnv, data.host));
  jEnv->SetObjectField(jInstance, info.fields["host"], jHost);

  jEnv->SetIntField(jInstance, info.fields["viewerCount"], static_cast<jint>(data.viewerCount));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityRaiding(
  JNIEnv* jEnv, const DashboardActivityRaiding& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityRaiding(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jRaider, GetJavaInstance_DashboardActivityUser(jEnv, data.raider));
  jEnv->SetObjectField(jInstance, info.fields["raider"], jRaider);

  jEnv->SetIntField(jInstance, info.fields["viewerCount"], static_cast<jint>(data.viewerCount));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivitySubscription(
  JNIEnv* jEnv, const DashboardActivitySubscription& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivitySubscription(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jSubscriber, GetJavaInstance_DashboardActivityUser(jEnv, data.subscriber));
  jEnv->SetObjectField(jInstance, info.fields["subscriber"], jSubscriber);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTier, GetJavaInstance_String(jEnv, data.tier));
  jEnv->SetObjectField(jInstance, info.fields["tier"], jTier);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivityResubscriptionSharing(
  JNIEnv* jEnv, const DashboardActivityResubscriptionSharing& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivityResubscriptionSharing(jEnv);
  JavaClassInfo& fragementInfo = GetJavaClassInfo_DashboardActivityFragment(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jSubscriber, GetJavaInstance_DashboardActivityUser(jEnv, data.subscriber));
  jEnv->SetObjectField(jInstance, info.fields["subscriber"], jSubscriber);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTier, GetJavaInstance_String(jEnv, data.tier));
  jEnv->SetObjectField(jInstance, info.fields["tier"], jTier);

  jEnv->SetIntField(jInstance, info.fields["cumulativeTenureMonths"], static_cast<jint>(data.cumulativeTenureMonths));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jCustomMessage, GetJavaInstance_String(jEnv, data.customMessage));
  jEnv->SetObjectField(jInstance, info.fields["customMessage"], jCustomMessage);

  // index required so no range based for-loop
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jCustomMessageFragments,
    jEnv->NewObjectArray(static_cast<jsize>(data.customMessageFragments.size()), fragementInfo.klass, nullptr));

  for (int i = 0; i < data.customMessageFragments.size(); i++) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jCustomMessageFragment,
      GetJavaInstance_DashboardActivityFragment(jEnv, data.customMessageFragments[i]));
    jEnv->SetObjectArrayElement(jCustomMessageFragments, i, jCustomMessageFragment);
  }

  jEnv->SetObjectField(jInstance, info.fields["customMessageFragments"], jCustomMessageFragments);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivitySubscriptionGiftingCommunity(
  JNIEnv* jEnv, const DashboardActivitySubscriptionGiftingCommunity& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivitySubscriptionGiftingCommunity(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jGifter, GetJavaInstance_DashboardActivityUser(jEnv, data.gifter));
  jEnv->SetObjectField(jInstance, info.fields["gifter"], jGifter);

  jEnv->SetBooleanField(jInstance, info.fields["anonymous"], (data.anonymous ? JNI_TRUE : JNI_FALSE));

  jEnv->SetIntField(jInstance, info.fields["quantity"], static_cast<jint>(data.quantity));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTier, GetJavaInstance_String(jEnv, data.tier));
  jEnv->SetObjectField(jInstance, info.fields["tier"], jTier);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_DashboardActivitySubscriptionGiftingIndividual(
  JNIEnv* jEnv, const DashboardActivitySubscriptionGiftingIndividual& data) {
  JavaClassInfo& info = GetJavaClassInfo_DashboardActivitySubscriptionGiftingIndividual(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jHeader, GetJavaInstance_DashboardActivityHeader(jEnv, data.header));
  jEnv->SetObjectField(jInstance, info.fields["header"], jHeader);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jGifter, GetJavaInstance_DashboardActivityUser(jEnv, data.gifter));
  jEnv->SetObjectField(jInstance, info.fields["gifter"], jGifter);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jRecipient, GetJavaInstance_DashboardActivityUser(jEnv, data.recipient));
  jEnv->SetObjectField(jInstance, info.fields["recipient"], jRecipient);

  jEnv->SetBooleanField(jInstance, info.fields["anonymous"], (data.anonymous ? JNI_TRUE : JNI_FALSE));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTier, GetJavaInstance_String(jEnv, data.tier));
  jEnv->SetObjectField(jInstance, info.fields["tier"], jTier);

  return jInstance;
}
