/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/generated/java_all.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/generated/jni_library.h"
#include "twitchsdk/core/java_coreregistries.h"
#include "twitchsdk/core/java_coreutil.h"
#include "twitchsdk/core/java_eventtrackerproxy.h"
#include "twitchsdk/core/java_httprequest.h"
#include "twitchsdk/core/java_socket.h"
#include "twitchsdk/core/java_tracerproxy.h"
#include "twitchsdk/core/systemclock.h"

using namespace ttv;
using namespace ttv::binding::java;

namespace ttv {
TTV_ErrorCode SetClientId(const std::string& clientId);
}

namespace {
using SocketFactoryList = std::vector<std::shared_ptr<JavaSocketFactory>>;
using WebSocketFactoryList = std::vector<std::shared_ptr<JavaWebSocketFactory>>;

// NOTE: These are smart pointers since putting vectors in the global context will cause memory allocation before the
// allocators are set
std::shared_ptr<SocketFactoryList> gJavaSocketFactories;
std::shared_ptr<WebSocketFactoryList> gJavaWebSocketFactories;
}  // namespace

// Forward declares from the auto-generated library.h
TTV_ErrorCode TTV_InitializeLibrary();
TTV_ErrorCode TTV_ShutdownLibrary();
const char* TTV_GetVersionString();

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_Initialize(JNIEnv* jEnv, jobject /*jThis*/) {
  CacheJavaVirtualMachine(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);

  TTV_ErrorCode ec = TTV_InitializeLibrary();

  JniThreadInitialize();

#if !NDEBUG
  if (TTV_SUCCEEDED(ec)) {
    ttv::trace::SetComponentMessageLevel("bindings", MessageLevel::Debug);
  }
#endif

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_Shutdown(JNIEnv* jEnv, jobject /*jThis*/) {
  TTV_ErrorCode ec = TTV_ShutdownLibrary();

  // Shutting down the socket library will clear out factory lists so make sure we do the same here
  if (TTV_SUCCEEDED(ec)) {
    gJavaSocketFactories.reset();
    gJavaWebSocketFactories.reset();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_Library_GetVersionString(JNIEnv* jEnv, jobject /*jThis*/) {
  const char* version = TTV_GetVersionString();

  return GetJavaInstance_String(jEnv, version);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetClientId(JNIEnv* jEnv, jobject /*jThis*/, jstring jClientId) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jClientId, TTV_EC_INVALID_ARG);

  ScopedJavaUTFStringConverter clientId(jEnv, jClientId);

  TTV_ErrorCode ec = ttv::SetClientId(clientId.GetNativeString());
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetComponentMessageLevel(
  JNIEnv* jEnv, jobject /*jThis*/, jstring jComponentName, jint jLevel) {
  ScopedJavaUTFStringConverter componentName(jEnv, jComponentName);
  MessageLevel traceLevel = static_cast<MessageLevel>(jLevel);

  TTV_ErrorCode ec = ttv::trace::SetComponentMessageLevel(componentName.GetNativeString(), traceLevel);
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetGlobalMessageLevel(JNIEnv* jEnv, jobject /*jThis*/, jint jLevel) {
  MessageLevel traceLevel = static_cast<MessageLevel>(jLevel);

  TTV_ErrorCode ec = ttv::trace::SetGlobalMessageLevel(traceLevel);
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetTraceLevel(JNIEnv* jEnv, jobject /*jThis*/, jint jLevel) {
  MessageLevel traceLevel = static_cast<MessageLevel>(jLevel);

  TTV_ErrorCode ec = ttv::trace::SetGlobalMessageLevel(traceLevel);
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_Library_ErrorToString(JNIEnv* jEnv, jobject /*jThis*/, jobject jErrorCode) {
  JavaClassInfo& info = GetJavaClassInfo_ErrorCode(jEnv);
  TTV_ErrorCode ec = static_cast<TTV_ErrorCode>(jEnv->CallIntMethod(jErrorCode, info.methods["getValue"]));

  const char* str = ttv::ErrorToString(ec);

  if (str == nullptr) {
    return nullptr;
  }

  return jEnv->NewStringUTF(str);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetHttpRequestProvider(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jIHttpRequestProvider) {
  // SEMI-HACK: Load classes here that we know will be needed by other native threads
  // See http://www.milk.com/kodebase/dalvik-docs-mirror/docs/jni-tips.html#FAQFindClass
  (void)GetJavaClassInfo_ISocket(jEnv);
  (void)GetJavaClassInfo_ISocketFactory(jEnv);
  (void)GetJavaClassInfo_IWebSocket(jEnv);
  (void)GetJavaClassInfo_IWebSocketFactory(jEnv);
  (void)GetJavaClassInfo_WebSocketMessageType(jEnv);
  (void)GetJavaClassInfo_ResultContainer(jEnv);
  (void)GetJavaClassInfo_IHttpRequestProvider(jEnv);
  (void)GetJavaClassInfo_HttpRequestResult(jEnv);
  (void)GetJavaClassInfo_HttpParameter(jEnv);
  (void)GetJavaClassInfo_ErrorCode(jEnv);
  (void)GetJavaClassInfo_String(jEnv);
  (void)GetJavaClassInfo_Charset(jEnv);
  (void)GetJavaClassInfo_Integer(jEnv);
  (void)GetJavaClassInfo_Boolean(jEnv);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (jIHttpRequestProvider != nullptr) {
    ttv::SetHttpRequest(std::make_shared<JavaHttpRequest>(jEnv, jIHttpRequestProvider));
  } else {
    ttv::SetHttpRequest(nullptr);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT void JNICALL Java_tv_twitch_Library_SetTracer(JNIEnv* jEnv, jobject /*jThis*/, jobject jTracer) {
  if (jTracer != nullptr) {
    auto tracer = std::make_shared<ttv::binding::java::JavaTracerProxy>();
    tracer->SetListener(jTracer);
    ttv::SetTracer(tracer);
  } else {
    ttv::SetTracer(nullptr);
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_SetEventTracker(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jEventTracker) {
  if (jEventTracker != nullptr) {
    auto eventTracker = std::make_shared<ttv::binding::java::JavaEventTrackerProxy>();
    eventTracker->SetListener(jEventTracker);
    ttv::SetEventTracker(eventTracker);
  } else {
    ttv::SetEventTracker(nullptr);
  }
  return GetJavaInstance_ErrorCode(jEnv, TTV_EC_SUCCESS);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_RegisterSocketFactory(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jFactory) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (jFactory == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  // Make sure it's not already registered
  if (TTV_SUCCEEDED(ec)) {
    if (gJavaSocketFactories == nullptr) {
      gJavaSocketFactories = std::make_shared<SocketFactoryList>();
    }

    for (auto iter = gJavaSocketFactories->begin(); iter != gJavaSocketFactories->end(); ++iter) {
      const auto& factory = *iter;
      jobject jOther = factory->GetJavaInstance().GetInstance();

      if (jEnv->IsSameObject(jOther, jFactory)) {
        ec = TTV_EC_INVALID_ARG;
        break;
      }
    }
  }

  // Register it
  if (TTV_SUCCEEDED(ec)) {
    auto factory = std::make_shared<JavaSocketFactory>(jEnv, jFactory);
    gJavaSocketFactories->push_back(factory);
    RegisterSocketFactory(factory);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_UnregisterSocketFactory(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jFactory) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (jFactory == nullptr || gJavaSocketFactories == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = TTV_EC_INVALID_ARG;

    for (auto iter = gJavaSocketFactories->begin(); iter != gJavaSocketFactories->end(); ++iter) {
      const auto& factory = *iter;
      jobject jOther = factory->GetJavaInstance().GetInstance();

      if (jEnv->IsSameObject(jOther, jFactory)) {
        ec = UnregisterSocketFactory(factory);
        TTV_ASSERT(TTV_SUCCEEDED(ec));
        gJavaSocketFactories->erase(iter);
        if (gJavaSocketFactories->size() == 0) {
          gJavaSocketFactories.reset();
        }
        break;
      }
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_RegisterWebSocketFactory(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jFactory) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (jFactory == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  // Make sure it's not already registered
  if (TTV_SUCCEEDED(ec)) {
    if (gJavaWebSocketFactories == nullptr) {
      gJavaWebSocketFactories = std::make_shared<WebSocketFactoryList>();
    }

    for (auto iter = gJavaWebSocketFactories->begin(); iter != gJavaWebSocketFactories->end(); ++iter) {
      auto factory = *iter;
      jobject jOther = factory->GetJavaInstance().GetInstance();

      if (jEnv->IsSameObject(jOther, jFactory)) {
        ec = TTV_EC_INVALID_ARG;
        break;
      }
    }
  }

  // Register it
  if (TTV_SUCCEEDED(ec)) {
    auto factory = std::make_shared<JavaWebSocketFactory>(jEnv, jFactory);
    gJavaWebSocketFactories->push_back(factory);
    RegisterWebSocketFactory(factory);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_Library_UnregisterWebSocketFactory(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jFactory) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (jFactory == nullptr || gJavaWebSocketFactories == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = TTV_EC_INVALID_ARG;

    for (auto iter = gJavaWebSocketFactories->begin(); iter != gJavaWebSocketFactories->end(); ++iter) {
      auto factory = *iter;
      jobject jOther = factory->GetJavaInstance().GetInstance();

      if (jEnv->IsSameObject(jOther, jFactory)) {
        ec = UnregisterWebSocketFactory(factory);
        TTV_ASSERT(TTV_SUCCEEDED(ec));
        gJavaWebSocketFactories->erase(iter);
        if (gJavaWebSocketFactories->size() == 0) {
          gJavaWebSocketFactories.reset();
        }
        break;
      }
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

// SDK-754 this API doesn't work, but it is unused on Android
JNIEXPORT jobject JNICALL Java_tv_twitch_Library_CreateBackgroundEventScheduler(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jJniThreadValidator) {
  std::shared_ptr<IEventScheduler> eventScheduler;
  TTV_ErrorCode ec = CreateBackgroundEventScheduler(eventScheduler);

  if (TTV_SUCCEEDED(ec)) {
    JavaClassInfo& proxyInfo = GetJavaClassInfo_EventSchedulerProxy(jEnv);

    // SDK-754 seems to crash here
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jProxy,
      jEnv->NewObject(proxyInfo.klass, proxyInfo.methods["<init>"], eventScheduler.get(), jJniThreadValidator));

    gBackgroundEventSchedulerRegistry.Register(eventScheduler, {}, jProxy);

    JavaClassInfo& successResultInfo = GetJavaClassInfo_SuccessResult(jEnv);
    jobject jSuccessResult = jEnv->NewObject(successResultInfo.klass, successResultInfo.methods["<init>"], jProxy);

    return jSuccessResult;
  } else {
    return GetJavaInstance_ErrorResult(jEnv, ec);
  }
}

JNIEXPORT jlong JNICALL Java_tv_twitch_Library_GetSystemClockTime(JNIEnv* /*jEnv*/, jobject /*jThis*/) {
  return static_cast<jlong>(GetSystemClockTime());
}

JNIEXPORT jlong JNICALL Java_tv_twitch_Library_MsToSystemTime(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jMilliseconds) {
  return static_cast<jlong>(MsToSystemTime(static_cast<uint64_t>(jMilliseconds)));
}
