/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/generated/jni_eventschedulerproxy.h"

#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/generated/java_all.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/java_coreregistries.h"
#include "twitchsdk/core/java_utility.h"

using namespace ttv;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IEventScheduler*>(x)

JNIEXPORT jobject JNICALL Java_tv_twitch_EventSchedulerProxy_ScheduleTask(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jTaskParams) {
  auto scheduler = GET_NATIVE_PTR(jNativePointer);

  TTV_RETURN_ON_NULL(scheduler, GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_INSTANCE));
  TTV_RETURN_ON_NULL(jTaskParams, GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG));

  JavaClassInfo& taskParamsInfo = GetJavaClassInfo_TaskParams(jEnv);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTaskName, jEnv->GetObjectField(jTaskParams, taskParamsInfo.fields["taskName"]));

  std::string taskName;
  if (jTaskName != nullptr) {
    ScopedJavaUTFStringConverter nameConverter(jEnv, jTaskName);
    taskName = nameConverter.GetNativeString();
  }

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jTaskFunction, jEnv->GetObjectField(jTaskParams, taskParamsInfo.fields["taskFunction"]));

  TTV_RETURN_ON_NULL(jTaskFunction, GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG));

  auto globalFunctionRef = std::make_shared<GlobalJavaObjectReference>();
  globalFunctionRef->Bind(jEnv, jTaskFunction);

  jlong delayMilliseconds = jEnv->GetLongField(jTaskParams, taskParamsInfo.fields["delayMilliseconds"]);
  if (delayMilliseconds < 0) {
    return GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG);
  }

  Result<TaskId> result = scheduler->ScheduleTask(
    {[globalFunctionRef = std::move(globalFunctionRef)]() {
       AutoJEnv jAutoEnv;

       JavaClassInfo& functionInfo = GetJavaClassInfo_TaskFunction(jAutoEnv);
       jAutoEnv->CallVoidMethod(globalFunctionRef->GetInstance(), functionInfo.methods["invoke"]);
     },
      static_cast<uint64_t>(delayMilliseconds), taskName});

  return GetJavaInstance_Result(jEnv, result, [jEnv](TaskId taskId) { return GetJavaInstance_TaskId(jEnv, taskId); });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_EventSchedulerProxy_CancelTask(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jTaskId) {
  auto scheduler = GET_NATIVE_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, scheduler, TTV_EC_INVALID_INSTANCE);
  TTV_JNI_RETURN_ON_NULL(jEnv, jTaskId, TTV_EC_INVALID_ARG);

  JavaClassInfo& taskIdInfo = GetJavaClassInfo_TaskId(jEnv);
  jlong taskId = jEnv->GetLongField(jTaskId, taskIdInfo.fields["id"]);

  if (taskId <= 0) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  TTV_ErrorCode ec = scheduler->CancelTask(static_cast<TaskId>(taskId));

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_EventSchedulerProxy_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jShutdownFunction) {
  auto scheduler = GET_NATIVE_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, scheduler, TTV_EC_INVALID_INSTANCE);
  TTV_JNI_RETURN_ON_NULL(jEnv, jShutdownFunction, TTV_EC_INVALID_ARG);

  auto globalFunctionRef = std::make_shared<GlobalJavaObjectReference>();
  globalFunctionRef->Bind(jEnv, jShutdownFunction);

  TTV_ErrorCode ec = scheduler->Shutdown([globalFunctionRef = std::move(globalFunctionRef)]() {
    AutoJEnv jAutoEnv;

    JavaClassInfo& functionInfo = GetJavaClassInfo_TaskFunction(jAutoEnv);
    jAutoEnv->CallVoidMethod(globalFunctionRef->GetInstance(), functionInfo.methods["invoke"]);
  });

  return GetJavaInstance_ErrorResult(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_EventSchedulerProxy_GetState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  auto scheduler = GET_NATIVE_PTR(jNativePointer);

  TTV_RETURN_ON_NULL(scheduler, nullptr);

  return GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_EventSchedulerState(jEnv), scheduler->GetState());
}

JNIEXPORT void JNICALL Java_tv_twitch_EventSchedulerProxy_DisposeNativeInstance(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  auto scheduler = GET_NATIVE_PTR(jNativePointer);

  if (scheduler != nullptr) {
    gBackgroundEventSchedulerRegistry.Unregister(scheduler);
  }
}
