package tv.twitch.test;

import java.util.*;
import tv.twitch.*;

public class CoreTest extends TestBase {
    protected CoreAPI m_CoreApi = null;
    protected boolean m_CoreApiInitialized = false;

    protected class CoreApiState { public boolean initialized = false; }

    protected class SimpleCounter { public int count = 0; }

    protected ICoreAPIListener m_CoreApiListener = new ICoreAPIListener() {
        @Override
        public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
            m_CoreApiInitialized = state == ModuleState.Initialized || state == ModuleState.ShuttingDown;
        }

        @Override
        public void coreUserLoginComplete(String oauthToken, int userId, ErrorCode ec) {}

        @Override
        public void coreUserLogoutComplete(int userId, ErrorCode ec) {}

        @Override
        public void coreUserAuthenticationIssue(int userId, String oauthToken, ErrorCode ec) {}

        @Override
        public void corePubSubStateChanged(int userId, CorePubSubState state, ErrorCode result) {}
    };

    /**
     * These Java tests by default uses WinInetHttpRequest::SendHttpRequest instead of JavaHttpRequest::SendHttpRequest
     * If you'd like to test JavaHttpRequest::SendHttpRequest, you can call the following:
     *      m_Library.setHttpRequestProvider(m_HttpRequestProvider);
     */
    private tv.twitch.IHttpRequestProvider m_HttpRequestProvider = new tv.twitch.IHttpRequestProvider() {
        @Override
        public ErrorCode sendHttpRequest(String requestName, String url, HttpParameter[] headerParams,
            byte[] requestBody, String httpReqType, int timeOutInSecs, HttpRequestResult result) {
            result.statusCode = 200;
            result.headers = new HttpParameter[1];
            result.headers[0] = new HttpParameter();
            result.headers[0].name = "accept";
            result.headers[0].value = "all things";
            result.response = "{}";
            return CoreErrorCode.TTV_EC_SUCCESS;
        }
    };

    // These are native functions (called in c++) that will call all of the functions in the listeners for test
    // purposes.
    private native void Test_ICoreAPIListener(IModule module, ICoreAPIListener listener);
    private native void Test_IChannelListener(IChannelListener listener);
    private native void Test_IEventTracker();
    private native void Test_ITracer();
    private native void Test_IDashboardActivityListener(IDashboardActivityListener listener);
    private native void Test_IGenericSubscriberListener(IGenericSubscriberListener listener);

    private native Result<String> Test_ReceiveSuccessResult();
    private native Result<String> Test_ReceiveErrorResult();
    private native void Test_SendSuccessResult(Result<String> stringResult);
    private native void Test_SendErrorResult(Result<String> stringResult);

    public CoreTest(Library library) {
        super(library);

        CoreErrorCode.forceClassInit();
    }

    @Override
    public void run() throws Exception {
        System.out.println("Running CoreTest tests...");

        // Uncomment these if you want to test
        // test_ICoreAPIListener();
        // test_IChannelListener();
        // test_IChannelStatus();
        // test_JavaHttpRequest();
        // test_IEventTracker();
        // test_IEventTrackerThreadSafety();
        // test_Result();
        // test_ITracer();
        // test_EventScheduler();
        // test_IDashboardActivityStatus();
        // test_IDashboardActivityListener();
        // test_IGenericSubscriberStatus();
        // test_IGenericSubscriberListener();
        // test_GetSubscribedPubsubTopics();
        System.out.println("Done running CoreTest tests...");
    }

    protected ErrorCode setup(final ResultContainer<CoreAPI> coreApi) throws InterruptedException {
        final CoreApiState coreApiState = new CoreApiState();

        ErrorCode ec = CoreErrorCode.TTV_EC_SUCCESS;

        ICoreAPIListener coreApiListener = new ICoreAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                coreApiState.initialized = state == ModuleState.Initialized || state == ModuleState.ShuttingDown;
            }

            @Override
            public void coreUserLoginComplete(String oauthToken, int userId, ErrorCode ec) {}

            @Override
            public void coreUserLogoutComplete(int userId, ErrorCode ec) {}

            @Override
            public void coreUserAuthenticationIssue(int userId, String oauthToken, ErrorCode ec) {}

            @Override
            public void corePubSubStateChanged(int userId, CorePubSubState state, ErrorCode result) {}
        };

        coreApi.result = new CoreAPI();
        coreApi.result.setListener(coreApiListener);
        initializeModule(coreApi.result);
        addModule(coreApi.result);

        // Log in a user
        if (m_OauthToken != null) {
            final ResultContainer<Boolean> callbackCalled = new ResultContainer<Boolean>();
            callbackCalled.result = false;

            ec = coreApi.result.logIn(m_OauthToken, new CoreAPI.LogInCallback() {
                @Override
                public void invoke(ErrorCode ec, UserInfo userInfo) {
                    callbackCalled.result = true;
                    if (ec.succeeded()) {
                        System.out.println("Logged in successfully");
                    }
                }
            });

            if (ec.succeeded()) {
                while (!callbackCalled.result) {
                    Thread.sleep(100);
                    updateModules();
                }
            }
        }

        return ec;
    }

    public void test_ICoreAPIListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final ICoreAPIListener coreApiListener = new ICoreAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void coreUserLoginComplete(String oauthToken, int userId, ErrorCode ec) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void coreUserLogoutComplete(int userId, ErrorCode ec) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void coreUserAuthenticationIssue(int userId, String oauthToken, ErrorCode ec) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void corePubSubStateChanged(int userId, CorePubSubState state, ErrorCode result) {
                calls.add(getCurrentMethodName());
            }
        };

        // Ask native to call each method on our interface
        Test_ICoreAPIListener(mDummyModule, coreApiListener);

        // Make sure they were all called
        checkAllMethodsCalled(ICoreAPIListener.class, calls);
    }

    public void test_IChannelListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final IChannelListener channelListener = new IChannelListener() {
            @Override
            public void streamUp(int playDelaySeconds) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamDown() {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamViewerCountChanged(int viewerCount) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamTriggeredMidroll(int durationSeconds) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamReceivedWatchPartyUpdate(WatchPartyUpdate update) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void profileImageUpdated(ProfileImage[] images) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamInfoUpdated(StreamInfoUpdate info) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void squadUpdated(SquadInfo info) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void squadLeft() {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void pixelTrackingUpdate(Boolean refresh) {
                calls.add(getCurrentMethodName());
            }
        };

        Test_IChannelListener(channelListener);

        checkAllMethodsCalled(IChannelListener.class, calls);
    }

    public void test_ITracer() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final ITracer tracer = new ITracer() {
            @Override
            public void log(String component, String messageLevel, String log) {
                calls.add(getCurrentMethodName());
            }
        };

        m_Library.setTracer(tracer);
        Test_ITracer();

        checkAllMethodsCalled(ITracer.class, calls);
        m_Library.setTracer(null);
    }

    public void test_IEventTracker() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final IEventTracker tracker = new IEventTracker() {
            @Override
            public ErrorCode trackEvent(String eventName, HashMap<String, Object> properties) {
                calls.add(getCurrentMethodName());
                return CoreErrorCode.TTV_EC_SUCCESS;
            }
        };

        m_Library.setEventTracker(tracker);
        Test_IEventTracker();

        checkAllMethodsCalled(IEventTracker.class, calls);
        m_Library.setEventTracker(null);
    }

    public void test_IEventTrackerThreadSafety() throws Exception {
        final SimpleCounter callCount = new SimpleCounter();

        final IEventTracker tracker = new IEventTracker() {
            @Override
            public ErrorCode trackEvent(String eventName, HashMap<String, Object> properties) {
                callCount.count++;
                return CoreErrorCode.TTV_EC_SUCCESS;
            }
        };

        m_Library.setEventTracker(tracker);
        Runnable task = new Runnable() {
            @Override
            public void run() {
                Test_IEventTracker();
            }
        };

        Thread[] threads = new Thread[5]; // new ArrayList<Thread>(5, task);
        Arrays.fill(threads, new Thread(task));

        for (int i = 0; i < 5; i++) {
            threads[i] = new Thread(task);
        }

        for (Thread thread : threads) {
            thread.start();
        }

        for (Thread thread : threads) {
            thread.join();
        }

        if (callCount.count != threads.length) {
            throw new Exception("test_IEventTrackerThreadSafety call_count " + java.lang.String.valueOf(callCount.count)
                + " :" + java.lang.String.valueOf(threads.length));
        }

        m_Library.setEventTracker(null);
    }

    public void test_IChannelStatus() throws Exception {
        ResultContainer<IChannelStatus> channelContainer = new ResultContainer<IChannelStatus>();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        setup(coreApiResult);

        CoreAPI coreApi = coreApiResult.result;

        try {
            // Test CoreApi::CreateChannelStatus
            final ResultContainer<Boolean> callbackCalled = new ResultContainer<Boolean>();
            callbackCalled.result = false;

            coreApi.createChannelStatus(m_UserId, m_UserId, null, channelContainer);

            byte[] bytes = {(byte) 137, 80, 78, 71, 13, 10, 26, 10}; // The start sequence for a PNG
            channelContainer.result.uploadProfileImage(
                bytes, bytes.length, new IChannelStatus.UploadProfileImageCallback() {
                    @Override
                    public void invoke(ErrorCode ec, ProfileImage[] images) {
                        callbackCalled.result = true;
                        return;
                    }
                });

            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }
        } finally {
            if (coreApi != null) {
                coreApi.shutdown(null);
                while (coreApi.getState() != ModuleState.Uninitialized) {
                    Thread.sleep(250);
                    updateModules();
                }
                coreApi.dispose();
            }

            if (channelContainer != null && channelContainer.result != null) {
                ChannelStatusProxy proxy = (ChannelStatusProxy) channelContainer.result;
                proxy.dispose();
            }
            clearModules();
        }
    }

    // Tests JavaHttpRequest::SendHttpRequest by using ChannelStatus::UploadProfileImage to test GET and POST requests
    public void test_JavaHttpRequest() throws Exception {
        ResultContainer<IChannelStatus> channelContainer = new ResultContainer<IChannelStatus>();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        setup(coreApiResult);

        CoreAPI coreApi = coreApiResult.result;

        try {
            final ResultContainer<Boolean> callbackCalled = new ResultContainer<Boolean>();
            callbackCalled.result = false;

            coreApi.createChannelStatus(m_UserId, m_UserId, null, channelContainer);

            byte[] bytes = {(byte) 137, 80, 78, 71, 13, 10, 26, 10}; // The start sequence for a PNG

            final tv.twitch.IHttpRequestProvider mockHttpRequestProvider = new tv.twitch.IHttpRequestProvider() {
                @Override
                public ErrorCode sendHttpRequest(String requestName, String url, HttpParameter[] headerParams,
                    byte[] requestBody, String httpReqType, int timeOutInSecs, HttpRequestResult result) {
                    String getUploadUrlEndpoint = "https://api.twitch.tv/kraken/users/" + m_UserId + "/upload_image";
                    if (url.equals(getUploadUrlEndpoint)) {
                        result.statusCode = 200;
                        result.response =
                            "{\"upload_url\":\"https://prod-web-upload-service-ingest.s3-accelerate.amazonaws.com/\",\"upload_id\":\"f61b6216-f2a6-45ca-a8d5-8f5613d88093\"}";
                    } else if (url.equals("https://prod-web-upload-service-ingest.s3-accelerate.amazonaws.com/")) {
                        // So that we don't start listening on a pub-sub topic to check whether the upload succeeded or
                        // not (we know it's not going to succeed).
                        result.statusCode = 401;

                        if (requestBody.length == bytes.length) {
                            for (int i = 0; i < requestBody.length; i++) {
                                if (requestBody[i] != bytes[i]) {
                                    callbackCalled.result = false;
                                    return CoreErrorCode.TTV_EC_SUCCESS;
                                }
                            }
                            callbackCalled.result = true;
                        }
                    }

                    return CoreErrorCode.TTV_EC_SUCCESS;
                }
            };

            m_Library.setHttpRequestProvider(mockHttpRequestProvider);

            final ResultContainer<Boolean> pubSubFired = new ResultContainer<Boolean>();
            pubSubFired.result = false;

            channelContainer.result.uploadProfileImage(
                bytes, bytes.length, new IChannelStatus.UploadProfileImageCallback() {
                    @Override
                    public void invoke(ErrorCode ec, ProfileImage[] images) {
                        pubSubFired.result = true;
                        return;
                    }
                });

            while (!callbackCalled.result && !pubSubFired.result) {
                Thread.sleep(250);
                updateModules();
            }

        } finally {
            if (coreApi != null) {
                coreApi.shutdown(null);
                while (coreApi.getState() != ModuleState.Uninitialized) {
                    Thread.sleep(250);
                    updateModules();
                }
                coreApi.dispose();
            }

            if (channelContainer != null && channelContainer.result != null) {
                ChannelStatusProxy proxy = (ChannelStatusProxy) channelContainer.result;
                proxy.dispose();
            }
            clearModules();
        }
    }

    public void test_Result() {
        Result<String> receiveSuccessResult = Test_ReceiveSuccessResult();
        assert (receiveSuccessResult.isSuccess());
        assert (!receiveSuccessResult.isError());
        assert (receiveSuccessResult.getErrorCode().equals(CoreErrorCode.TTV_EC_SUCCESS));
        assert (receiveSuccessResult.getResult().equals("hello world?"));

        Result<String> receiveErrorResult = Test_ReceiveErrorResult();
        assert (!receiveErrorResult.isSuccess());
        assert (receiveErrorResult.isError());
        assert (receiveSuccessResult.getErrorCode().equals(CoreErrorCode.TTV_EC_UNSUPPORTED));
        assert (receiveSuccessResult.getResult() == null);

        Result<String> sendSuccessResult = new SuccessResult<String>("hello world!");
        Test_SendSuccessResult(sendSuccessResult);

        Result<String> sendErrorResult = new ErrorResult<String>(CoreErrorCode.TTV_EC_UNIMPLEMENTED);
        Test_SendErrorResult(sendErrorResult);
    }

    public void test_EventScheduler() throws Exception {
        Result<IEventScheduler> result = m_Library.createBackgroundEventScheduler();
        assert (result.isSuccess());

        final IEventScheduler scheduler = result.getResult();

        scheduler.scheduleTask(new TaskParams(new TaskFunction() {
            @Override
            public void invoke() {
                System.out.println("Task 1 (immediate)");
            }
        }));

        scheduler.scheduleTask(new TaskParams(new TaskFunction() {
            @Override
            public void invoke() {
                System.out.println("Task 2 (delayed)");
                scheduler.scheduleTask(new TaskParams(new TaskFunction() {
                    @Override
                    public void invoke() {
                        System.out.println("Task 3 (chained)");
                    }
                }, "ChainedTask"));
            }
        }, 200));

        scheduler.scheduleTask(new TaskParams(new TaskFunction() {
            @Override
            public void invoke() {
                System.out.println("Task 4 (immediate)");
            }
        }, 0, "ImmediateTask"));

        Thread.sleep(500);

        scheduler.shutdown(new TaskFunction() {
            @Override
            public void invoke() {
                System.out.println("Shut down scheduler.");
            }
        });

        Thread.sleep(100);

        scheduler.dispose();
    }

    public void test_IDashboardActivityListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final IDashboardActivityListener listener = new IDashboardActivityListener() {
            @Override
            public void eventBitsUseage(DashboardActivityBitsUsage data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventFollower(DashboardActivityFollow data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventAutoHostStart(DashboardActivityHost data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventHostStart(DashboardActivityHost data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventRaiding(DashboardActivityRaiding data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventSubscription(DashboardActivitySubscription data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventPrimeSubscription(DashboardActivitySubscription data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventResubscriptionSharing(DashboardActivityResubscriptionSharing data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventPrimeResubscriptionSharing(DashboardActivityResubscriptionSharing data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventSubscriptionGiftingCommunity(DashboardActivitySubscriptionGiftingCommunity data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventSubscriptionGiftingIndividual(DashboardActivitySubscriptionGiftingIndividual data) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void eventStreamUp(DashboardActivityHeader data) {
                calls.add(getCurrentMethodName());
            }
        };

        Test_IDashboardActivityListener(listener);

        checkAllMethodsCalled(IDashboardActivityListener.class, calls);
    }

    public void test_IDashboardActivityStatus() throws Exception {
        ResultContainer<IDashboardActivityStatus> dashboardActivityContainer =
            new ResultContainer<IDashboardActivityStatus>();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        setup(coreApiResult);

        CoreAPI coreApi = coreApiResult.result;

        coreApi.createDashboardActivityStatus(m_UserId, 54321, null, dashboardActivityContainer);

        if (coreApi != null) {
            coreApi.shutdown(null);
            while (coreApi.getState() != ModuleState.Uninitialized) {
                Thread.sleep(250);
                updateModules();
            }
            coreApi.dispose();
        }

        if (dashboardActivityContainer != null && dashboardActivityContainer.result != null) {
            DashboardActivityStatusProxy proxy = (DashboardActivityStatusProxy) dashboardActivityContainer.result;
            proxy.dispose();
        }
        clearModules();
    }

    public void test_IGenericSubscriberListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        final IGenericSubscriberListener listener = new IGenericSubscriberListener() {
            @Override
            public void eventTopicData(String data) {
                calls.add(getCurrentMethodName());
            }
        };

        Test_IGenericSubscriberListener(listener);

        checkAllMethodsCalled(IGenericSubscriberListener.class, calls);
    }

    public void test_IGenericSubscriberStatus() throws Exception {
        ResultContainer<IGenericSubscriberStatus> genericSubscriberContainer =
            new ResultContainer<IGenericSubscriberStatus>();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        setup(coreApiResult);

        CoreAPI coreApi = coreApiResult.result;

        coreApi.createGenericSubscriberStatus(m_UserId, "test_topic.12345", null, genericSubscriberContainer);

        if (coreApi != null) {
            coreApi.shutdown(null);
            while (coreApi.getState() != ModuleState.Uninitialized) {
                Thread.sleep(250);
                updateModules();
            }
            coreApi.dispose();
        }

        if (genericSubscriberContainer != null && genericSubscriberContainer.result != null) {
            GenericSubscriberStatusProxy proxy = (GenericSubscriberStatusProxy) genericSubscriberContainer.result;
            proxy.dispose();
        }
        clearModules();
    }

    public void test_GetSubscribedPubsubTopics() throws Exception {
        ResultContainer<IGenericSubscriberStatus> genericSubscriberContainer =
            new ResultContainer<IGenericSubscriberStatus>();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        setup(coreApiResult);

        CoreAPI coreApi = coreApiResult.result;

        coreApi.createGenericSubscriberStatus(0, "ad-property-refresh.12345", null, genericSubscriberContainer);

        String[] topics = new String[0];
        long stopTime = System.currentTimeMillis() + 1000;

        while (!Arrays.asList(topics).contains("ad-property-refresh.12345") && stopTime > System.currentTimeMillis()) {
            Thread.sleep(1);
            updateModules();
            ResultContainer<String[]> result = new ResultContainer<>();
            ErrorCode ec = coreApi.getSubscribedPubsubTopics(result);
            if (ec != CoreErrorCode.TTV_EC_SUCCESS) {
                throw new Exception("getSubscribedPubsubTopics Failed " + ec.getName());
            }
            topics = result.result;
        }

        if (!Arrays.asList(topics).contains("ad-property-refresh.12345")) {
            throw new Exception("getSubscribedPubsubTopics Cannot find topic ");
        }

        if (coreApi != null) {
            coreApi.shutdown(null);
            while (coreApi.getState() != ModuleState.Uninitialized) {
                Thread.sleep(250);
                updateModules();
            }
            coreApi.dispose();
        }

        if (genericSubscriberContainer != null && genericSubscriberContainer.result != null) {
            GenericSubscriberStatusProxy proxy = (GenericSubscriberStatusProxy) genericSubscriberContainer.result;
            proxy.dispose();
        }
        clearModules();
    }
}
