/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/java_coretest.h"

#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/java_channellistenerproxy.h"
#include "twitchsdk/core/java_coreapilistenerproxy.h"
#include "twitchsdk/core/java_coreutil.h"
#include "twitchsdk/core/java_dashboardactivitylistenerproxy.h"
#include "twitchsdk/core/java_genericsubscriberlistenerproxy.h"

using namespace ttv;
using namespace ttv::binding::java;

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1ICoreAPIListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jModule, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaCoreAPIListenerProxy>(jModule);
    listener->SetListener(jListener);

    listener->ModuleStateChanged(nullptr, IModule::State::Initialized, TTV_EC_SUCCESS);
    listener->CoreUserLoginComplete("oauthToken", 9001, ErrorDetails(TTV_EC_SUCCESS));
    listener->CoreUserLogoutComplete(9001, TTV_EC_SUCCESS);
    listener->CoreUserAuthenticationIssue(9001, "oauthToken", ErrorDetails(TTV_EC_SUCCESS));
    listener->CorePubSubStateChanged(9001, PubSubState::Connected, TTV_EC_SUCCESS);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1IChannelListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaChannelListenerProxy>();
    listener->SetListener(jListener);

    listener->StreamUp(30);
    listener->StreamDown();
    listener->StreamViewerCountChanged(1337);
    listener->StreamTriggeredMidroll(15);
    listener->ProfileImageUpdated(std::vector<ProfileImage>());
    listener->PixelTrackingUpdate(true);

    WatchPartyUpdate update;
    update.incrementUrl = "http://increment.com";
    update.vodId = "v12345";
    update.title = "Stream title";
    update.watchPartyId = "WatchPartyId12345";
    update.broadcastType = VodType::Archive;
    update.viewable = true;

    listener->StreamReceivedWatchPartyUpdate(update);

    StreamInfoUpdate info;
    info.title = "Stream title";
    info.game = "Dark Souls";
    info.gameId = 1000;

    listener->StreamInfoUpdated(std::move(info));

    SquadMember member;
    member.channelId = 12345;
    member.userLogin = "twitch";
    member.userDisplayName = "Twitch";
    member.profileImageUrl150 = "profileimageurl.jpg";

    SquadInfo squad;
    squad.ownerId = 12826;
    squad.squadId = "squadid";
    squad.status = SquadStatus::Live;
    squad.members.push_back(member);

    listener->SquadUpdated(std::move(squad));
    listener->SquadLeft();
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1IEventTracker(JNIEnv* /*jEnv*/, jobject /*jThis*/) {
  ttv::TrackEvent("TestTrackingEvent", {{"test_null", nullptr}, {"test_boolean", true}, {"test_integer", 777},
                                         {"test_double", 3.0}, {"test_string", "stringValue"}});
}

JNIEXPORT jobject JNICALL Java_tv_twitch_test_CoreTest_Test_1ReceiveSuccessResult(JNIEnv* jEnv, jobject /*jThis*/) {
  auto result = MakeSuccessResult(std::string{"hello world?"});

  jobject jResult = GetJavaInstance_Result(
    jEnv, result, [jEnv](const std::string& string) { return GetJavaInstance_String(jEnv, string); });

  return jResult;
}

JNIEXPORT jobject JNICALL Java_tv_twitch_test_CoreTest_Test_1ReceiveErrorResult(JNIEnv* jEnv, jobject /*jThis*/) {
  Result<std::string> result = MakeErrorResult(TTV_EC_UNSUPPORTED);

  jobject jResult = GetJavaInstance_Result(
    jEnv, result, [jEnv](const std::string& string) { return GetJavaInstance_String(jEnv, string); });

  return jResult;
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1SendSuccessResult(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jResult) {
  auto result = GetNativeInstance_Result<std::string>(jEnv, jResult, [jEnv](jobject jString) {
    return ScopedJavaUTFStringConverter(jEnv, static_cast<jstring>(jString)).GetNativeString();
  });

  TTV_ASSERT(result.IsSuccess());
  TTV_ASSERT(!result.IsError());
  TTV_ASSERT(TTV_SUCCEEDED(result.GetErrorCode()));
  TTV_ASSERT(result.GetResult() == "hello world!");
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1SendErrorResult(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jResult) {
  auto result = GetNativeInstance_Result<std::string>(jEnv, jResult, [jEnv](jobject jString) {
    return ScopedJavaUTFStringConverter(jEnv, static_cast<jstring>(jString)).GetNativeString();
  });

  TTV_ASSERT(!result.IsSuccess());
  TTV_ASSERT(result.IsError());
  TTV_ASSERT(result.GetErrorCode() == TTV_EC_UNIMPLEMENTED);
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1ITracer(JNIEnv*, jobject) {
  MessageLevel oldMessageLevel = MessageLevel::None;
  ttv::trace::GetGlobalMessageLevel(oldMessageLevel);
  ttv::trace::SetGlobalMessageLevel(MessageLevel::Error);
  ttv::trace::Message("coretest", MessageLevel::Error, "This is a test error message to trigger a tracer callback");
  ttv::trace::SetGlobalMessageLevel(oldMessageLevel);
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1IDashboardActivityListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaDashboardActivityListenerProxy>();
    listener->SetListener(jListener);

    DashboardActivityBitsUsage dashboardActivityBitsUsage;
    dashboardActivityBitsUsage.header.id = "1";
    dashboardActivityBitsUsage.header.timestamp = 2;
    dashboardActivityBitsUsage.user.displayName = "3";
    dashboardActivityBitsUsage.user.login = "4";
    dashboardActivityBitsUsage.user.userId = "5";
    dashboardActivityBitsUsage.anonymous = false;
    dashboardActivityBitsUsage.amount = 6;

    listener->EventBitsUseage(dashboardActivityBitsUsage);

    DashboardActivityFollow dashboardActivityFollow;
    dashboardActivityFollow.header.id = "7";
    dashboardActivityFollow.header.timestamp = 8;
    dashboardActivityFollow.follower.displayName = "9";
    dashboardActivityFollow.follower.login = "10";
    dashboardActivityFollow.follower.userId = "11";

    listener->EventFollower(dashboardActivityFollow);

    DashboardActivityHost dashboardActivityHost;
    dashboardActivityHost.header.id = "12";
    dashboardActivityHost.header.timestamp = 13;
    dashboardActivityHost.host.displayName = "14";
    dashboardActivityHost.host.login = "15";
    dashboardActivityHost.host.userId = "16";
    dashboardActivityHost.viewerCount = 17;

    listener->EventAutoHostStart(dashboardActivityHost);

    dashboardActivityHost.header.id = "18";
    dashboardActivityHost.header.timestamp = 19;
    dashboardActivityHost.host.displayName = "20";
    dashboardActivityHost.host.login = "21";
    dashboardActivityHost.host.userId = "22";
    dashboardActivityHost.viewerCount = 23;

    listener->EventHostStart(dashboardActivityHost);

    DashboardActivityRaiding dashboardActivityRaiding;
    dashboardActivityRaiding.header.id = "24";
    dashboardActivityRaiding.header.timestamp = 25;
    dashboardActivityRaiding.raider.displayName = "26";
    dashboardActivityRaiding.raider.login = "27";
    dashboardActivityRaiding.raider.userId = "28";
    dashboardActivityRaiding.viewerCount = 29;

    listener->EventRaiding(dashboardActivityRaiding);

    DashboardActivitySubscription dashboardActivitySubscription;
    dashboardActivitySubscription.header.id = "30";
    dashboardActivitySubscription.header.timestamp = 31;
    dashboardActivitySubscription.subscriber.displayName = "32";
    dashboardActivitySubscription.subscriber.login = "33";
    dashboardActivitySubscription.subscriber.userId = "34";
    dashboardActivitySubscription.tier = "35";

    listener->EventSubscription(dashboardActivitySubscription);

    dashboardActivitySubscription.header.id = "36";
    dashboardActivitySubscription.header.timestamp = 37;
    dashboardActivitySubscription.subscriber.displayName = "38";
    dashboardActivitySubscription.subscriber.login = "39";
    dashboardActivitySubscription.subscriber.userId = "40";
    dashboardActivitySubscription.tier = "41";

    listener->EventPrimeSubscription(dashboardActivitySubscription);

    DashboardActivityResubscriptionSharing dashboardActivityResubscriptionSharing;
    dashboardActivityResubscriptionSharing.header.id = "42";
    dashboardActivityResubscriptionSharing.header.timestamp = 43;
    dashboardActivityResubscriptionSharing.subscriber.displayName = "44";
    dashboardActivityResubscriptionSharing.subscriber.login = "45";
    dashboardActivityResubscriptionSharing.subscriber.userId = "46";
    dashboardActivityResubscriptionSharing.tier = "47";
    dashboardActivityResubscriptionSharing.cumulativeTenureMonths = 48;
    dashboardActivityResubscriptionSharing.customMessage = "49";
    dashboardActivityResubscriptionSharing.customMessageFragments.push_back({"1", "2", "3"});

    listener->EventResubscriptionSharing(dashboardActivityResubscriptionSharing);

    dashboardActivityResubscriptionSharing.header.id = "50";
    dashboardActivityResubscriptionSharing.header.timestamp = 51;
    dashboardActivityResubscriptionSharing.subscriber.displayName = "52";
    dashboardActivityResubscriptionSharing.subscriber.login = "53";
    dashboardActivityResubscriptionSharing.subscriber.userId = "54";
    dashboardActivityResubscriptionSharing.tier = "55";
    dashboardActivityResubscriptionSharing.cumulativeTenureMonths = 56;
    dashboardActivityResubscriptionSharing.customMessage = "57";
    dashboardActivityResubscriptionSharing.customMessageFragments.push_back({"4", "5", "6"});

    listener->EventPrimeResubscriptionSharing(dashboardActivityResubscriptionSharing);

    DashboardActivitySubscriptionGiftingCommunity dashboardActivitySubscriptionGiftingCommunity;
    dashboardActivitySubscriptionGiftingCommunity.header.id = "58";
    dashboardActivitySubscriptionGiftingCommunity.header.timestamp = 59;
    dashboardActivitySubscriptionGiftingCommunity.gifter.displayName = "60";
    dashboardActivitySubscriptionGiftingCommunity.gifter.login = "61";
    dashboardActivitySubscriptionGiftingCommunity.gifter.userId = "62";
    dashboardActivitySubscriptionGiftingCommunity.quantity = 63;
    dashboardActivitySubscriptionGiftingCommunity.tier = "64";

    listener->EventSubscriptionGiftingCommunity(dashboardActivitySubscriptionGiftingCommunity);

    DashboardActivitySubscriptionGiftingIndividual dashboardActivitySubscriptionGiftingIndividual;
    dashboardActivitySubscriptionGiftingIndividual.header.id = "65";
    dashboardActivitySubscriptionGiftingIndividual.header.timestamp = 66;
    dashboardActivitySubscriptionGiftingIndividual.gifter.displayName = "67";
    dashboardActivitySubscriptionGiftingIndividual.gifter.login = "68";
    dashboardActivitySubscriptionGiftingIndividual.gifter.userId = "69";
    dashboardActivitySubscriptionGiftingIndividual.recipient.displayName = "70";
    dashboardActivitySubscriptionGiftingIndividual.recipient.login = "71";
    dashboardActivitySubscriptionGiftingIndividual.recipient.userId = "72";
    dashboardActivitySubscriptionGiftingIndividual.tier = "73";

    listener->EventSubscriptionGiftingIndividual(dashboardActivitySubscriptionGiftingIndividual);

    DashboardActivityHeader dashboardActivityHeader;
    dashboardActivityHeader.id = "74";
    dashboardActivityHeader.timestamp = 75;
    listener->EventStreamUp(dashboardActivityHeader);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_CoreTest_Test_1IGenericSubscriberListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaGenericSubscriberListenerProxy>();
    listener->SetListener(jListener);
    std::string data = R"({"string": "test_string",)"
                       R"("timestamp": "2019-04-01T04:10:10Z",)"
                       R"("double": 0.004,)"
                       R"("array": [)"
                       R"(  {)"
                       R"(    "obj1": {)"
                       R"(      "obj1_1": "1",)"
                       R"(      "obj1_2": "2")"
                       R"(    })"
                       R"(  },{)"
                       R"(    "obj2": {)"
                       R"(      "obj2_1": "1",)"
                       R"(      "obj2_2": "2")"
                       R"(    })"
                       R"(  })"
                       R"(]})";
    listener->EventTopicData(data);
  }
}
