package tv.twitch;

import java.util.HashMap;
import java.util.Map;

public abstract class ErrorCode {
    /**
     * The global map of error value to the instance.
     */
    private static Map<Integer, ErrorCode> s_ValueMap = new HashMap<>();
    /**
     * The global map of error name to the instance.
     */
    private static Map<String, ErrorCode> s_NameMap = new HashMap<>();
    /**
     * Returns the registered ErrorCode instance or null if not registered.
     */
    public static ErrorCode lookupValue(int value) {
        synchronized (s_ValueMap) { return s_ValueMap.get(value); }
    }
    /**
     * Returns the registered ErrorCode instance or null if not registered.
     */
    public static ErrorCode lookupName(String name) {
        synchronized (s_NameMap) { return s_NameMap.get(name); }
    }

    private int m_Value;
    private String m_Name;

    protected ErrorCode(int value, String name) {
        this.m_Value = value;
        this.m_Name = name;

        // Register the instance
        synchronized (s_ValueMap) { s_ValueMap.put(value, this); }
        synchronized (s_NameMap) { s_NameMap.put(name, this); }
    }

    public int getValue() { return m_Value; }

    public String getName() { return m_Name; }

    public boolean succeeded() { return m_Value == 0; }

    public boolean failed() { return m_Value != 0; }

    @Override
    public boolean equals(Object other) {
        // Check against another ErrorCode
        if (other != null && other instanceof ErrorCode) {
            ErrorCode otherErrorCode = (ErrorCode) other;
            return m_Value == otherErrorCode.m_Value;
        }
        // Check against an integer
        else if (other != null && other instanceof Integer) {
            Integer otherInt = (Integer) other;
            return m_Value == otherInt.intValue();
        }
        // Default
        else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        return m_Value;
    }
}
