package tv.twitch;

/**
 * This class acts as a Java proxy for an object allocated natively.  The native instance can be obtained in 2 ways:
 * - A concrete class overrides createNativeInstance() and returns a valid instance.
 * - An explicit instance to wrap is passed into the constructor.
 */
public abstract class NativeProxy {
    private long mNativeObjectPointer = 0;
    protected final IJniThreadValidator mJniThreadValidator;

    protected NativeProxy(IJniThreadValidator jniThreadValidator) {
        if (jniThreadValidator == null) {
            throw new IllegalArgumentException("jniThreadValidator must not be null");
        }
        mJniThreadValidator = jniThreadValidator;
        mNativeObjectPointer = createNativeInstance();
    }

    protected NativeProxy(long nativeObjectPointer, IJniThreadValidator jniThreadValidator) {
        if (jniThreadValidator == null) {
            throw new IllegalArgumentException("jniThreadValidator must not be null");
        }
        mNativeObjectPointer = nativeObjectPointer;
        mJniThreadValidator = jniThreadValidator;
    }

    /**
     * Creates a new native instance.  The default implementation does not allocate anything.  This method must be
     * overridden by concrete types that wish to allow allocations that are initiated by Java constructors.
     *
     * @return The native pointer or 0 if the allocation fails or is not allowed.
     */
    protected long createNativeInstance() { return 0; }

    /**
     * To be implemented by concrete implementations to perform the actual native disposal.
     */
    protected abstract void disposeNativeInstance(long nativeObjectPointer);

    protected long getNativeObjectPointer() { return mNativeObjectPointer; }

    @Override
    protected void finalize() throws Throwable {
        dispose();
    }

    /**
     * Explicitly disposes the native object so it can be done predictably.  Relying on the garbage collector to do so
     * could cause leaks or crashes depending on the object.
     */
    public void dispose() {
        if (mNativeObjectPointer != 0) {
            disposeNativeInstance(mNativeObjectPointer);
            mNativeObjectPointer = 0;
        }
    }
}
