package tv.twitch;

import java.util.concurrent.atomic.AtomicReference;

/**
 * Checks that only one thread at a time calls SDK JNI methods
 */
public final class SingleJniThreadChecker implements IJniThreadChecker {
    private AtomicReference<Thread> currentJniThreadAtomicReference = new AtomicReference<Thread>();

    @Override
    public void beforeJni() {
        final Thread currentThread = Thread.currentThread();
        if (!currentJniThreadAtomicReference.compareAndSet(null, currentThread)) {
            // We might be unlucky here and not get the comparison that actually failed
            // This is just a best effort. It might be null, or it might be an entirely different thread
            final Thread currentJniThread = currentJniThreadAtomicReference.get();
            throw new IllegalStateException("SDK JNI calls should only happen on one thread at a time, "
                + "but a JNI call is probably happening on " + currentJniThread + " and you tried to call another on "
                + currentThread);
        }
    }

    @Override
    public void afterJni() {
        currentJniThreadAtomicReference.compareAndSet(Thread.currentThread(), null);
    }
}
