/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/java_classinfo.h"

#include "twitchsdk/core/assertion.h"

bool ttv::binding::java::LookupJavaClass(JNIEnv* jEnv, JavaClassInfo& info, const char* klass) {
  TTV_ASSERT(jEnv);
  TTV_ASSERT(klass != nullptr);

  jclass cls = jEnv->FindClass(klass);
  if (cls == nullptr) {
    ttv::trace::Message("bindings", MessageLevel::Error, "FindClass failed: %s", klass);
  } else {
    ttv::trace::Message("bindings", MessageLevel::Debug, "FindClass succeeded: %s", klass);

    // get a global reference to ensure the class isn't unloaded
    info.klass = static_cast<jclass>(jEnv->NewGlobalRef(cls));

    jEnv->DeleteLocalRef(cls);
  }

  TTV_ASSERT(cls != nullptr);

  return info.klass != nullptr;
}

bool ttv::binding::java::LookupJavaMethod(
  JNIEnv* jEnv, JavaClassInfo& info, const char* method, const char* signature) {
  TTV_ASSERT(jEnv);
  TTV_ASSERT(method != nullptr);
  TTV_ASSERT(signature != nullptr);

  jmethodID mid = jEnv->GetMethodID(info.klass, method, signature);
  if (mid == nullptr) {
    ttv::trace::Message("bindings", MessageLevel::Error, "GetMethodID failed: %s %s", method, signature);
  } else {
    ttv::trace::Message("bindings", MessageLevel::Debug, "GetMethodID succeeded: %s %s", method, signature);

    info.methods[method] = mid;
  }

  TTV_ASSERT(mid != nullptr);

  return mid != nullptr;
}

bool ttv::binding::java::LookupJavaStaticMethod(
  JNIEnv* jEnv, JavaClassInfo& info, const char* method, const char* signature) {
  TTV_ASSERT(jEnv);
  TTV_ASSERT(method != nullptr);
  TTV_ASSERT(signature != nullptr);

  jmethodID mid = jEnv->GetStaticMethodID(info.klass, method, signature);
  if (mid == nullptr) {
    ttv::trace::Message("bindings", MessageLevel::Error, "GetStaticMethodID failed: %s %s", method, signature);
  } else {
    ttv::trace::Message("bindings", MessageLevel::Debug, "GetStaticMethodID succeeded: %s %s", method, signature);

    info.staticMethods[method] = mid;
  }

  TTV_ASSERT(mid != nullptr);

  return mid != nullptr;
}

bool ttv::binding::java::LookupJavaField(JNIEnv* jEnv, JavaClassInfo& info, const char* field, const char* signature) {
  TTV_ASSERT(jEnv);
  TTV_ASSERT(field != nullptr);
  TTV_ASSERT(signature != nullptr);

  jfieldID fid = jEnv->GetFieldID(info.klass, field, signature);
  if (fid == nullptr) {
    ttv::trace::Message("bindings", MessageLevel::Error, "GetFieldID failed: %s %s", field, signature);
  } else {
    ttv::trace::Message("bindings", MessageLevel::Debug, "GetFieldID succeeded: %s %s", field, signature);

    info.fields[field] = fid;
  }

  TTV_ASSERT(fid != nullptr);

  return fid != nullptr;
}
