/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/java_httprequest.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/httprequestutils.h"

ttv::binding::java::JavaHttpRequest::JavaHttpRequest(JNIEnv* jEnv, jobject jObject) {
  mJavaInstance.Bind(jEnv, jObject);
}

TTV_ErrorCode ttv::binding::java::JavaHttpRequest::SendHttpRequest(const std::string& requestName,
  const std::string& url, const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody,
  size_t requestBodySize, HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData) {
  if (url.size() == 0) {
    return TTV_EC_INVALID_HTTP_REQUEST_PARAMS;
  }

  if (!responseCallback) {
    return TTV_EC_INVALID_HTTP_REQUEST_PARAMS;
  }

  TTV_ASSERT(mJavaInstance.GetInstance() != nullptr);
  if (mJavaInstance.GetInstance() == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  TTV_ErrorCode ec = TTV_EC_API_REQUEST_FAILED;

  // Use a scope to ensure references are released before the thread is detached
  AutoJEnv jEnv;
  {
    // Convert native parameters into Java objects
    JavaClassInfo& providerInfo = GetJavaClassInfo_IHttpRequestProvider(jEnv);

    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRequestName, GetJavaInstance_String(jEnv, requestName));
    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUrl, GetJavaInstance_String(jEnv, url));
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jRequestHeaders, GetJavaInstance_HttpParameterArray(jEnv, requestHeaders));
    AUTO_DELETE_LOCAL_REF(jEnv, jbyteArray, jRequestBody, jEnv->NewByteArray(static_cast<jsize>(requestBodySize)));
    jEnv->SetByteArrayRegion(
      jRequestBody, 0, static_cast<jsize>(requestBodySize), reinterpret_cast<const jbyte*>(requestBody));

    jstring jHttpReqType = nullptr;
    switch (httpReqType) {
      case HTTP_PUT_REQUEST: {
        jHttpReqType = GetJavaInstance_String(jEnv, "PUT");
        break;
      }
      case HTTP_POST_REQUEST: {
        jHttpReqType = GetJavaInstance_String(jEnv, "POST");
        break;
      }
      case HTTP_DELETE_REQUEST: {
        jHttpReqType = GetJavaInstance_String(jEnv, "DELETE");
        break;
      }
      case HTTP_GET_REQUEST:
      default: {
        jHttpReqType = GetJavaInstance_String(jEnv, "GET");
        break;
      }
    }
    AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jstring, jHttpReqType);

    // Allocate the object that will contain the result of the request
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jRequestResult, GetJavaInstance_HttpRequestResult(jEnv));

    // Call into Java to make the request synchronously
    ttv::trace::Message("bindings", MessageLevel::Debug, "Calling into Java IHttpRequestProvider implementation...");
    jobject jErrorCode =
      jEnv->CallObjectMethod(mJavaInstance.GetInstance(), providerInfo.methods["sendHttpRequest"], jRequestName, jUrl,
        jRequestHeaders, jRequestBody, jHttpReqType, static_cast<jint>(timeOutInSecs), jRequestResult);
    ttv::trace::Message("bindings", MessageLevel::Debug, "Java IHttpRequestProvider implementation returned");

    AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobject, jErrorCode);

    // Convert the result into native types
    ec = GetNativeFromJava_SimpleEnum<TTV_ErrorCode>(
      jEnv, GetJavaClassInfo_ErrorCode(jEnv), jErrorCode, TTV_EC_UNKNOWN_ERROR);

    // If the call succeeds then fire callbacks
    if (TTV_SUCCEEDED(ec)) {
      uint statusCode = 0;
      std::map<std::string, std::string> resultHeaders;
      std::vector<char> response;
      GetNativeInstance_HttpRequestResult(jEnv, jRequestResult, statusCode, resultHeaders, response);

      // Notify the client
      bool notify = true;
      if (headersCallback != nullptr) {
        notify = headersCallback(statusCode, resultHeaders, userData);
      }

      if (notify) {
        responseCallback(statusCode, response, userData);
      }
    }
  }

  ttv::trace::Message("bindings", MessageLevel::Debug, "Done processing HTTP response from Java");

  return ec;
}
